/* Copyright (c)1994-1999 Begemot Computer Associates. All rights reserved.
 * See the file COPYRIGHT for details of redistribution and use. */

# include "proc.h"

/*
 * generic device support
 */

# define TINTERVAL	20	/* msecs between clock ticks */


typedef struct Async	Async;
typedef struct Timer	Timer;
typedef struct Timeout	Timeout;

/*
 * structure to register external device driver programs.
 * Each external program has a pipe on fd 0 to p11.
 * P11 is notified by an SIGIO if one of the pipes has something
 * to read. It selects all pipe fds and calls the device's async routine
 * for each ready file descriptor.
 */
struct Async {
	IODev	*dev;		/* device that registered asynchonuous notification */
	void	*argp;		/* call argument */
	pid_t	pid;		/* external driver pid */
};

/*
 * struct to describe the state of the line clock.
 */
struct Timer {
	int	tinterval;	/* repeat interval */
	int	trunning;	/* timer is running */
};

/*
 * struct for timeout table.
 */
struct Timeout {
	void	(*func)(void *);/* function to be called */
	void	*arg;		/* call argument */
	uint	time;		/* interval, 0 means disabled */
	uint	curr;		/* rest in current interval */
};

fd_set	fd_devices;		/* pipes to external drivers */
Async	async[FD_DEVICES];	/* external driver table */
int	asy_init;		/* true if initialized */

Timeout	timeouts[MAXTIMEOUT];	/* timeout table */
int	ntimeouts;		/* current number of registered timeouts */

Timer	timer = { TINTERVAL };	/* line clock */


static void	onsigio();
static void	onalarm();
static void	kill_children(void);
static void	dev_pcmd(IODevCmd *tab, int pos);
static void	dev_pusage(IODevCmd *tab, int pos);
static void	set_fd_async(int fd);


static void	block(sigset_t *, int);
static void	unblock(sigset_t *);



/*
 * catch signals used by device support (SIGALRM and SIGIO)
 */
void
catch_io_sigs(void)
{
	catch_signal(SIGIO, onsigio);
	catch_signal(SIGALRM, onalarm);
}


/*
 * functions to block/unblock signals during global data manipulation
 */
static void
block(sigset_t *oset, int sig)
{
	sigset_t set;

	sigemptyset(&set);
	sigaddset(&set, sig);
	if(sigprocmask(SIG_BLOCK, &set, oset))
		panic("sigprocmask(SIG_BLOCK) failed: %s", strerror(errno));
}

static void
unblock(sigset_t *oset)
{
	if(sigprocmask(SIG_SETMASK, oset, NULL))
		panic("sigprocmask(SIG_SETMASK) failed: %s", strerror(errno));
}


static void
dotimer(struct timeval *oval)
{
	struct itimerval tval;

	tval.it_interval.tv_usec = 1000 * timer.tinterval;
	tval.it_interval.tv_sec = 0;

	if(oval != NULL && timerisset(oval))
		tval.it_value = *oval;
	else
		tval.it_value = tval.it_interval;

	if(setitimer(ITIMER_REAL, &tval, 0))
		panic("setitimer: %s", strerror(errno));
}

/*
 * start timer if it is not already running
 */
void
start_timer(void)
{

	if(timer.trunning)
		return;

	dotimer(NULL);

	timer.trunning = 1;
}

void
timer_faster()
{
	struct itimerval oval;

	if(timer.tinterval <= TINTERVAL - 2)
		return;

	timer.tinterval--;

	if(!timer.trunning)
		return;

	if(getitimer(ITIMER_REAL, &oval))
		panic("getitimer: %s", strerror(errno));

	dotimer(&oval.it_value);

# if 0
	printf("turned timer faster %d\n", timer.tinterval);
# endif
}

void
timer_slower()
{
	struct itimerval oval;

	if(timer.tinterval >= TINTERVAL + 2)
		return;

	timer.tinterval++;

	if(!timer.trunning)
		return;
	if(getitimer(ITIMER_REAL, &oval))
		panic("getitimer: %s", strerror(errno));

	dotimer(&oval.it_value);

# if 0
	printf("turned timer slower %d\n", timer.tinterval);
# endif
}

/*
 * stop timer if it is running
 * return old running state
 */
int
stop_timer(void)
{
	struct itimerval tval;

	if(!timer.trunning)
		return 0;

	tval.it_interval.tv_usec = tval.it_value.tv_usec = 0;
	tval.it_interval.tv_sec = tval.it_value.tv_sec = 0;
	setitimer(ITIMER_REAL, &tval, 0);
	timer.trunning = 0;

	return 1;
}


/*
 * this is the signal handler for the line clock, called each 20ms
 */
static void	
onalarm()
{
	Timeout *t;

	for(t = timeouts; t < &timeouts[ntimeouts]; t++)
		if(t->time && --t->curr == 0) {
			t->func(t->arg);
			t->curr = t->time;
		}
}

/*
 * Make a file descriptor generate a signal if input is ready
 */
static void
set_fd_async(int fd)
{
# ifdef HAVE_STREAMS
	if(ioctl(fd, I_SETSIG, S_RDNORM))
		panic("ioctl(I_SETSIG): %s", strerror(errno));
# else
# ifdef HAVE_O_ASYNC
	int fl;

	if(fcntl(fd, F_SETOWN, getpid()))
		panic( "fcntl(F_SETOWN): %s", strerror(errno) );
	if((fl = fcntl(fd, F_GETFL, 0)) == -1)
		panic( "fcntl(F_GETFL): %s", strerror(errno) );
	if(fcntl(fd, F_SETFL, O_ASYNC | fl))
		panic( "fcntl(F_SETFL): %s", strerror(errno) );
# else
# endif
# endif
}

/*
 * start an asynchronous device handler.
 * 
 *
 * there should be a method for the calling driver to
 * be informed when the external process dies.
 * Perhaps we could add a flag to the async function for this case.
 *
 * input:
 *	prog	executable to start
 *	argv	argument vector for new process
 *	dev	device to be informed on possible input
 *	argp	call argument for device's async function
 *	afd	file descriptor to retain open (or -1) (is duped to 3)
 *	pp	to return new process's pid (if not NULL)
 *
 * returns:
 *	pipe files descriptor
 */
int
register_handler(char *prog, char *argv[], IODev *dev, void *argp, int afd, pid_t *pp)
{
	int	sd[2];
	int	pid;
	int	fd;
	Async	*asy;
	sigset_t oset;
	char *full;

	block(&oset, SIGIO);

	if(!asy_init) {
		atexit(kill_children);
		asy_init = 1;
	}

	if(socketpair(PF_UNIX, SOCK_STREAM, 0, sd) < 0)
		panic("socketpair: %s", strerror(errno));

	switch(pid = fork()) {

	case -1:
		/* failure */
		panic("fork: %s", strerror(errno));
	case 0:
		/* child */
		(void)close(0);
		if(dup(sd[0]) != 0)
			abort();

		/*
		 * don't close fds 0 (it's the pipe), 2 (stderr) and
 		 * afd. We need afd because MMAP_INHERIT seems not to be
 		 * implemented so we need a shared file descriptor for
 		 * naming of the shared region (Urrgh, wouldn't it
		 * be _MUCH_ simpler to have plan9 rfork()?).
		 * dup afd to 3, so external program knows where to find it
		 */
		for(fd = getdtablesize() - 1; fd > 0; fd--)
			if(fd != 2 && fd != afd)
				close(fd);
		if(afd >= 0 && afd != 3) {
			if(dup2(afd, 3) < 0)
				panic("dup2(%d,%d): %s", afd, 3, strerror(errno));
			close(afd);
		}
		execv(prog, argv);
		if(iodir) {
			full = xalloc(strlen(prog) + strlen(iodir) + 2);
			sprintf(full, "%s/%s", iodir, prog);
			execv(full, argv);
		}
		panic("exec: %s - %s", prog, strerror(errno));

	}

	/* parent */
	close(sd[0]);

	asy = &async[sd[1]];
	asy->dev = dev;
	asy->argp = argp;
	asy->pid = pid;

	FD_SET(sd[1], &fd_devices);

	if(pp != NULL)
		*pp = pid;

	if(fcntl(sd[1], F_SETFL, O_NONBLOCK))
		panic( "fcntl(O_NONBLOCK): %s", strerror(errno) );

	set_fd_async(sd[1]);

	unblock(&oset);

	return sd[1];
}

/*
 * on exit kill all registered children
 * (but be honest and send a SIGTERM)
 */
static void
kill_children(void)
{
	Async	*asy;

	signal(SIGCHLD, SIG_IGN);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGIO, SIG_IGN);
	for(asy = async; asy < &async[FD_DEVICES]; asy++)
		if(asy->pid)
			kill(asy->pid, SIGTERM);
}


/*
 * signal handler for notification of possible i/o.
 * select all async pipes and call device handlers for all selected
 */
static void 
onsigio()
{
	fd_set	set = fd_devices;
	int	fd;
	Async	*asy;
	static struct timeval tv;

	if(select(FD_DEVICES+1, &set, 0, 0, &tv) > 0)
		for(fd = 0, asy = async; fd < FD_DEVICES; fd++, asy++)
			if(FD_ISSET(fd, &set))
				(*asy->dev->ioops->async)(asy->dev, asy->argp);
}


/*
 * register timer to be called each p msecs.
 * return index in timeout table
 */
int
register_timer(unsigned p, void (*f)(void *), void *a)
{
	Timeout	*t;
	sigset_t oset;
	int	i;

	block(&oset, SIGALRM);

	if(ntimeouts == MAXTIMEOUT)
		panic("out of timeouts");

	t = &timeouts[i = ntimeouts++];

	t->func = f;
	t->time = t->curr = p / timer.tinterval;
	t->arg = a;

	unblock(&oset);

	return i; 
}

/*
 * set timer to new period
 */
void
reset_timer(unsigned p, int i)
{
	Timeout *t;
	sigset_t oset;

	block(&oset, SIGALRM);

	t = &timeouts[i];
	t->time = t->curr = p / timer.tinterval;

	unblock(&oset);
}

/*
 * general command handler
 */
void
dev_command(IODev *dev, int argc, char **argv)
{
	IODevCmd	*p;
	int		matches;
	IODevCmd	*match = NULL;	/* keep compiler happy */
	IODevCmd	*tab = dev->ioops->cmds;

	matches = 0;
	for(p = tab; p->name != NULL; p++)
		if(SUBSTR(argv[0], p->name)) {
			match = p;
			matches++;
		}

	if(matches > 1) {
		printf("ambiguous command (%d matches):\n", matches);
		for(p = tab; p->name != NULL; p++)
			if(SUBSTR(argv[0], p->name))
				dev_pusage(tab, p - tab);
		return;

	} else if(matches == 0) {
		printf("Unknown command for '%s'; try '?'\n", dev->name);
		return;
	}

	if((*match->func)(dev, --argc, ++argv)) {
		printf("Usage:");
		dev_pusage(tab, match - tab);
	}
}

/*
 * print help info for device
 */
int
dev_help(IODev *dev, int argc, char **argv)
{
	IODevCmd	*p;
	int		matches;
	int		i;
	IODevCmd	*tab = dev->ioops->cmds;

	if(argc == 0) {
		/*
		 * list all
		 */
		printf("Available commands for device '%s':\n", dev->name);
		for(p = tab; p->name != NULL; p++)
			dev_pusage(tab, p - tab);
		return 0;
	}
	/*
	 * print matching commands for each argument
	 */
	for(i = 0; i < argc; i++) {
		for(matches = 0, p = tab; p->name != NULL; p++)
			if(SUBSTR(argv[i], p->name))
				matches++;
		if(matches == 0)
			printf("No match for '%s' for device '%s'.\n", argv[i], dev->name);
		else {
			printf("%d match%s for '%s' for device '%s':\n", matches, (matches > 1) ? "es" : "", argv[i], dev->name);
			for(p = tab; p->name != NULL; p++)
				if(SUBSTR(argv[i], p->name))
					dev_pusage(tab, p - tab);
		}
	}
	return 0;
}
/*
 * print command string with optional part in [] on stdout
 * do it the simple way: enlarge substring until we have only one
 * match left. This assumes the you have no command that is a substring
 * of a longer one, i.e. 'show' and 'showfile'.
 */
void
dev_pcmd(IODevCmd *tab, int pos)
{
	int		matches;
	IODevCmd	*p;
	u_int		len;

	for(len = 1; len < strlen(tab[pos].name); len++) {
		matches = 0;
		for(p = tab; p->name != NULL; p++)
			if(strlen(p->name) >= len && strncmp(p->name, tab[pos].name, len) == 0)
				matches++;
		if(matches == 1) {
			printf("%.*s[%s]", (int)len, tab[pos].name, tab[pos].name + len);
			return;
		}
	}
	printf("%s", tab[pos].name);
}

/*
 * print usage string on stdout
 */
void
dev_pusage(IODevCmd *tab, int pos)
{
	putchar('\t');
	dev_pcmd(tab, pos);
	printf(" %s\n", tab[pos].usage);
}

/*
 * Generica DMA handler
 * Transfers a number of bytes from a buffer into memory (or opposite).
 * Returns number of bytes transfered. If this is not the original amount,
 * a non-existant memory trap occured.
 */
uint
dma(void *buf, uint bytes, uint mem, DmaDir_t direction)
{
	uchar	*a;		/* memory address */
	uchar	*end_mem;	/* 1st illegal memory address */
	uchar	*end;		/* transfer end */

	end_mem = (uchar *)proc.memops->addrphys(proc.mmg, proc.physmem);

	a = (uchar *)proc.memops->addrphys(proc.mmg, mem);
	if(mem & 1)
		a++;

	end = a + bytes;
	if(end > end_mem)
		bytes -= (end - end_mem);

	if(bytes) {
		switch(direction) {

		case ReadMem:
			memcpy(buf, a, bytes);
			break;

		case WriteMem:
			memcpy(a, buf, bytes);
			break;

		default:
			panic("bad switch: %s, %d", __FILE__, __LINE__);
		}
	}

	return bytes;
}
