MODULE killer;

{This program plays the game of Killer supposedly invented by Bill Freeman.
4 players represented by 4 oblong boxes move about an 8 X 8 board, their
movements being controlled by the 4 button boxes.  If one player crashes into
another, then the 1st player gains a life (to a maximum of 9) and the 2nd
player loses a life.  If a player loses all his lives then he is retired from
the game and dissapears from the board.  When only one player remains, he is
the winner and is suitably congratulated.  Another game then begins.

The board is represented by a 10 X 10 array (8 X 8 board + a border to mark the
edge).  The players are numbered 1 2 3 4 and put onto the board by having their
numbers written into the square that they occupy.  The boarder is numbered -1
and free squares 0.  Each player also has a record called boxpos.  This gives
the x and y coordinates of that player at any time.  The players are displayed
by means of a record of 4 descriptors, 1 for each player.  This record
draws the player's box and writes his lives left onto the screen.

The players are moved by checking the button boxes each time the gt40 interrupts
and modifying the display files accordingly.  An array of booleans (alive)
indicates whether players ought to be moved.  When a player is killed
alive for him is set to false and playersalive is decremented.  If it reaches
1 then the last player is congratulated and the game is over.  A boolean
gameover is set to true and a timer set to count from 250 down to 0 in 1/50 ths
of a second.  The display handler (gt40) checks this counter and when it reaches
zero starts a new game.

C. Bond  26/02/78.
}
DEVICE MODULE control[4];

{This module provides the interface to the 4 button boxes which move the
players.  A player is moved 1 square in the selected direction only when a
button is depressed (ie to move 2 units in 1 direction, the button must be
depressed, released and depressed again).  The 4 procedures below thus only
read true when a button goes from up to down.  This is accomplished by storing
the last state of the buttons and comparing with the new.  The buttons for 1
player are read once only per cycle.}

DEFINE sensebuttons, left, right, up, down;

VAR camstr[160000B] : integer;  {station register}
    camcsr[160002B] : bits;     {status register}
    camxcr[160004B] : bits;     {executive register}
    camswr[162040B] : bits;     {switch module (button boxes)}
    oldswr, newswr : bits;      {look for change between these}
    r, l, u, d : ARRAY 1 : 4 OF integer; {for selecting bits in swr}

PROCEDURE sensebuttons;
BEGIN
  oldswr := newswr; newswr := camswr;
END sensebuttons;
PROCEDURE left (p : integer) : boolean;
BEGIN
  left := (NOT oldswr[l[p]]) AND (newswr[l[p]])
END left;

PROCEDURE right (p : integer) : boolean;
BEGIN
  right := (NOT oldswr[r[p]]) AND (newswr[r[p]])
END right;

PROCEDURE up (p : integer) : boolean;
BEGIN
  up := (NOT oldswr[u[p]]) AND (newswr[u[p]])
END up;

PROCEDURE down (p : integer) : boolean;
BEGIN
  down := (NOT oldswr[d[p]]) AND (newswr[d[p]])
END down;

BEGIN {control}
  {set up bit maps for swr}
  r[1] := 0; l[1] := 1; d[1] := 2; u[1] := 3;
  r[2] := 4; l[2] := 5; d[2] := 6; u[2] := 7;
  r[3] := 8; l[3] := 9; d[3] := 10; u[3] := 11;
  r[4] := 12; l[4] := 13; d[4] := 14; u[4] := 15;

  camstr := 5; camcsr := []; camxcr := [0, 7, 8, 14]; newswr := []
END control;
DEVICE MODULE timing[6];

{This module is used to time the pause between games.  Timer is called
with a number of clock ticks, the clock then counts the counter as set,
down to zero.  The calling process can check the counter and when it
reaches zero then it has paused long enough.}

DEFINE timer, counter;

VAR counter : integer;  {For calling process to check}
    on : boolean;  {Indicates to clock whether timer is running}

PROCEDURE timer (n : integer);
BEGIN
  counter := n; on := true
END timer;

PROCESS clock[100B];
  VAR lcsr[177546B] : bits;
BEGIN
  LOOP
    lcsr[6] := true; doio; lcsr[6] := false;
    IF on THEN
      dec (counter); IF counter = 0 THEN on := false END
    END
  END
END clock;

BEGIN {timing}
  counter := 0; on := false; clock
END timing;

DEVICE MODULE gt40[4];

{This module provides the interface to the display processor and controls the
game.  The button boxes are serviced and the display files modified accordingly
only when the display interrupts.}

DEFINE startgame, screen;

USE sensebuttons, left, right, up, down, timer, counter;

CONST pntmod = 117620B;  {Point mode}
      chrmod = 103620B;  {Character mode}
      lvhmod = 113524B;  {Long Vector solid(Heavy) line}
      lvlmod = 113525B;  {Long Vector Long dash line}
      lvsmod = 113526B;  {Long Vector Short dash line}
      lvdmod = 113527B;  {Long Vector Dot dashed line}
      dstop = 173400B;  {Display stop and interrupt}
      cw = 14;  {Character width}
      seeline = 40000B;  {intensify vector or point}
      negbit = 20000B;  {negative vector ie left or down}
      xlength = 200B;  {size of board (1 unit) in x direction}
      ylength = 140B;  {size of board (1 unti) in y direction}
      xdead = 9 * xlength;  {position off board for when dead (x)}
      ydead = 9 * ylength;  {position off board for when dead (y)}

TYPE direction = (l, r, u, d);  {directions for moving in}
     xypos = RECORD {to hold a players x y position on board}
             xpos : integer;
             ypos : integer
             END;
     boxpic = RECORD {display file for 1 box}
              lines : ARRAY 1 : 15 OF integer;
              lives : char
              END;

VAR alive : ARRAY 1 : 4 OF boolean;  {whether players are dead or alive}
    board : ARRAY 0 : 9, 0 : 9 OF integer;  {state of game on board}
    boxpos : ARRAY 1 : 4 OF xypos;  {positions of players on board}
    gameover : boolean;  {whether game is in progress or over}
    playersalive : integer;  {number of players still alive}
    i, j : integer;  {sundry variables}

    display : RECORD {players' boxes}
              body : ARRAY 1 : 4 OF boxpic;
              tail : integer
              END;
    wonmes : RECORD {congratulatory message holder}
             head : ARRAY 1 : 4 OF integer;
             body : ARRAY 1 : 47 OF char;
             tail : integer
             END;

PROCEDURE ring; {rings the bell in the gt40}
  VAR dsr[172002B] : integer;
BEGIN
  dsr := 1
END ring;

PROCEDURE startgame;
  VAR i, j : integer;
BEGIN
  gameover := false; playersalive := 4;
  alive := (true, true, true, true);

  {set up board : border to -1, other squares to 0}
  i := 0;
  REPEAT
    j := 0;
    REPEAT
      IF (i = 0) OR (i = 9) OR (j = 0) OR (j = 9) THEN
        board[i,j] := -1
      ELSE
        board[i,j] := 0
      END;
      inc (j)
    UNTIL j > 9;
    inc (i)
  UNTIL i > 9;

  {put players onto board}
  board[1,1] := 1; board[8,1] := 2;
  board[8,8] := 3; board[1,8] := 4;

  {set players' positions in their position descriptors}
  boxpos[1].xpos := 1; boxpos[1].ypos := 1;
  boxpos[2].xpos := 8; boxpos[2].ypos := 1;
  boxpos[3].xpos := 8; boxpos[3].ypos := 8;
  boxpos[4].xpos := 1; boxpos[4].ypos := 8;

  {reset graphics file for players}
  i := 1;
  REPEAT
    display.body[i].lines[2] := (boxpos[i].xpos - 1) * xlength;
    display.body[i].lines[3] := (boxpos[i].ypos - 1) * ylength;
    display.body[i].lives := '9';
    inc (i)
  UNTIL i > 4
END startgame;

PROCEDURE stopgame (p : integer); {player p has just won the game}
BEGIN
  gameover := true;
  IF boxpos[p].ypos >= 5 THEN {write message at bottom of screen}
    wonmes.head[3] := 2 * ylength
  ELSE {write message at top of screen}
    wonmes.head[3] := 6 * ylength
  END;
  wonmes.body[24] := char (p + 60B);
  timer (250)  {pause for 5 secs}
END stopgame;

PROCEDURE gainlife (p : integer); {increase player p's lives by 1 (max 9)}
  VAR l : integer;
BEGIN
  l := integer (display.body[p].lives);
  IF l <> integer ('9') THEN
    inc (l);
    display.body[p].lives := char (l)
  END
END gainlife;

PROCEDURE loselife (p, x, y : integer);

{Player p loses a life.  Check that he is not dead (if so remove him
from the board) and also that the game is not over.  If game is over find
out who has won and stop the game.}

  VAR l : integer;
BEGIN
  l := integer (display.body[p].lives);
  IF l = integer ('1') THEN {player is on his way out}
    alive[p] := false; {mark him dead}
    display.body[p].lines[2] := xdead;  {push him off display}
    display.body[p].lines[3] := ydead;
    board[x,y] := 0;  {and remove hom from the board}
    dec (playersalive);
    IF playersalive = 1 THEN {need to know who he is as he has won}
      IF alive[1] THEN stopgame (1)
      ELSIF alive[2] THEN stopgame (2)
      ELSIF alive[3] THEN stopgame (3)
      ELSIF alive[4] THEN stopgame (4)
      END
    END
  ELSE {player just loses a life}
    dec (l);
    display.body[p].lives := char (l)
  END;
  ring {make nice bleep noise in recognition of losing a life}
END loselife;

PROCEDURE movebox (x, y, nx, ny, p : integer);

{Attempt to move player p from x y to nx ny on board}

  VAR p1 : integer;  {what is on the new square}
BEGIN
  p1 := board[nx,ny];  {get details of new square}
  IF p1 = 0 THEN {square is empty - move there}
    board[nx,ny] := p; board[x,y] := 0;  {move him on board}
    boxpos[p].xpos := nx; boxpos[p].ypos := ny;  {put record of him straight}
    display.body[p].lines[2] := (nx - 1) * xlength; {put into display file}
    display.body[p].lines[3] := (ny - 1) * ylength;
  ELSIF p1 <> -1 THEN {we hit somebody !}
    gainlife (p); loselife (p1, nx, ny)
  ELSE {we hit the wall - how careless}
    loselife (p, x, y)
  END
END movebox;

PROCEDURE moveplayer (p : integer; dir : direction);

{Work out new position of player p who wishes to move and try to move him}

  VAR x, y : integer;   {old position}
      nx, ny : integer; {new position}
BEGIN
  x := boxpos[p].xpos; y := boxpos[p].ypos;  {get old position}
  nx := x; ny := y;
  IF dir = r THEN inc (nx)  {move to right}
  ELSIF dir = l THEN dec (nx)  {move to left}
  ELSIF dir = u THEN inc (ny)  {move up}
  ELSIF dir = d THEN dec (ny)  {move down}
  END;
  movebox (x, y, nx, ny, p)  {make the move}
END moveplayer;

PROCESS screen[320B];

{This process controls the gt40 itself and on interrupt checks the players
buttons and moves them accordingly.  If the game is over it maintains the
display (including the congratulatory message) until time is up
whereupon it initiates a new game.  The players' buttons are checked
in such a pattern such that the player to be checked first
changes every time.  A cyclic system is employed, the counter p
indicating which player is to be checked next and counter n making
sure that 4 players are checked each time.  P is incremented 5 times
per display interrupt, therefore the cycle moves on by 1 making a fair system.}

  VAR dpc[172000B] : integer;
  VAR p, n : integer;
BEGIN
  p := 0;
  LOOP
    IF NOT gameover THEN {service players that are alive}
      n := 1; {Set counter of players serviced this time}
      inc (p); {move on in cycle an extra player}
      IF p = 5 THEN p := 1 END;
      sensebuttons;  {see where they want to move}
      REPEAT
        IF alive [p] THEN
          IF left (p) THEN moveplayer (p, l)
          ELSIF right (p) THEN moveplayer (p, r)
          ELSIF up (p) THEN moveplayer (p, u)
          ELSIF down (p) THEN moveplayer (p, d)
          END
        END;
        inc (n); inc (p); {Move onto next player}
        IF p = 5 THEN p := 1 END
      UNTIL (n > 4) OR (gameover)
    ELSIF counter = 0 THEN startgame
    ELSE {maintain congratulatory message}
      dpc := adr (wonmes.head[1]);
      doio
    END;
    {draw the boxes}
    dpc := adr (display.body[1].lines[1]);
    doio
  END
END screen;

BEGIN {gt40}
  i := 1;
  REPEAT
    WITH display.body[i] DO
      lines := (pntmod, 0, 0, {for locating coordinates}
                0, {for line type}
                xlength - 3 + seeline, 0, {base of box}
                0 + seeline, ylength - 3, {right side}
                xlength - 3 + negbit + seeline, 0, {top of box}
                0 + seeline, ylength - 3 + negbit, {left side of box}
                0, 0, {for shifting to position where lives will be written}
                chrmod {for printing lives left});
    END;
    inc (i)
  UNTIL i > 4;

  WITH display DO
    body[1].lines[4] := lvhmod;
    body[2].lines[4] := lvlmod;
    body[3].lines[4] := lvsmod;
    body[4].lines[4] := lvdmod;
    {add shifts to put lives left in correct places for each box}
    body[1].lines[13] := 30; body[1].lines[14] := 25;
    body[2].lines[13] := 84; body[2].lines[14] := 25;
    body[3].lines[13] := 84; body[3].lines[14] := 50;
    body[4].lines[13] := 30; body[4].lines[14] := 50
  END;
  display.tail := dstop;

  {intialise the victory message}
  WITH wonmes DO
    head := (pntmod, 13 * cw, 0, {to be filled in} chrmod);
    body := 'Congratulations player X, you have won the game';
    tail := dstop
  END
END gt40;

BEGIN {killer}
  startgame;
  screen
END killer.
{
.bp
}
