/************************************************************************************************************
 *
 * Terminal Booking System (TBS).
 *				Michael Rourke - UNSW - January 1980
 *
 *	Fixed for Ed 7 and V32 by
 *				Chris Maltby - Sydney Uni - February 1980
 *
 *	Revised 
 *				Michael Rourke - UNSW - August 1980
 *
 ************************************************************************************************************/

#include "tbs.defines.h"

/* global vars returned by getargs */

bool		sday, stimerange, stime, suser, sttype, sforval, 
		stermrange, sterm;
int		day, timehi, timelo, user, ttype, forval, termhi, termlo;

/* passwd entry and strings - used by finduid, findlog, checkpw */

struct pwent	pe;
char		sbuf[40];

/* vars returned by book */

bool		hitrate, hitusage, nextbest, hitbooked, hitunusable,
		hitunbookable, hitcurrent;
int		fday, ftime, fterm;

/* vars returned by init */

int		bookfd, currentperiod, uid;
struct tm	*tim;
long		time(), systime, timezone, modtime();

/* vars used by evict */

char		*evictargs[NUMTERMS + 3];
char		**elistptr = &evictargs[1];

/* others */

bool		rewrite;		/* flag for getperiod */
char		obuf[BUFSIZ];		/* output buffer */
enum command	thiscommand;		/* the current tbs command */
usagetype	ausage;			/* returned by getusage */
bool		gotusage;		/* set if ausage is valid */
termtype	aterm;			/* returned by getterm */
header		head;			/* the booking file header */
int		iforval;		/* initial forval */

int		argc;			/* global argc, argv */
char		**argv;

/* action routines called by search routines (findfree, findused, sufindused) */
int		cbook(), book(), pfree(), list(), sulist(),
		unbook(), suunbook(), mkevictlist();

enum command	findname();
enum symbol	nextarg();

main(c, v)
int c;
char **v;
{
	argc = c;
	argv = v;
	init();
	if(getargs())
		exit(-1);
	thiscommand = findname();
	if(is(Cunbook) && not su)
		error("Permission denied");
	if(su)
		susetdef();
	else
		setdef();
	iforval = forval;
	switch(thiscommand)
	{
	case Book:
		if(findfree(day, timelo, termlo, (class ? cbook : book)) == 0)
		{
			fputs("No booking performed - no free periods\n", stdout);
			break;
		}
		flushprint();
		if(iforval == forval)
			fputs("No booking performed - ", stdout);
		else if(forval == 0)
			break;
		explain();
		break;
	case Free:
		if(findfree(day, timelo, termlo, pfree) == 0)
		{
			fputs("No free periods\n", stdout);
			break;
		}
		flushprint();
		if(forval == iforval)
			fputs("No free periods - ", stdout);
		else if(forval == 0 || not (sday && stime))
			break;
		explain();
		break;
	case List:
		if(not su || suser)
		{
			if(findused(day, timelo, termlo, user, list) && iforval != forval)
				break;
		} 
		else
			if(sufindused(day, timelo, termlo, sulist) && iforval != forval)
				break;
		flushprint();
		fputs("No periods booked ", stdout);
		if(sday || stime || sterm || sttype)
			fputs("in range specified\n", stdout);
		else
			fputs("\n", stdout);
		break;
	case Unbook:
	case Cunbook:
		if(not su || suser)
		{
			if(findused(day, timelo, termlo, user, unbook) && iforval != forval)
				break;
		} 
		else
			if(sufindused(day, timelo, termlo, suunbook) && iforval != forval)
				break;
		flushprint();
		fputs("No unbooking performed - ", stdout);
		if(hitcurrent)
			fputs("Can't unbook current period\n", stdout);
		else if(sday || stime || sterm || sttype)
			fputs("No booked periods in range specified\n", stdout);
		else
			fputs("No booked periods\n", stdout);
		break;
	case Evict:
		if(day != tim->dayofwk || timelo != currentperiod || timehi != timelo)
			error("Can only evict current period");
		if(findused(tim->dayofwk, currentperiod, termlo, user, mkevictlist) == 0 || evictargs[1] == 0)
			fputs("Can't Evict - not booked for current period\n", stdout);
		else
		{
			switch(fork())
			{
			case -1:
				error("Can't fork");
				break;
			case 0:
				/* child */
				signal(2, 1);
				signal(3, 1);
				evictargs[0] = "evict";
				*elistptr++ = pe.pw_strings[LNAME];
				*elistptr = 0;
				execv(evictprog, evictargs);
				perror(evictprog);
				exit(-1);
			default:
				/* parent */
				fputs("Eviction notice served\n", stdout);
				fflush(stdout);
				exit(0);
			}
		}
		break;
	default:
		snark("main");
	}
	flushprint();
	if((not su || suser) && not class)
	{
		if(not gotusage)
		{
			lockbookfile(READ);
			getusage(user);
			unlock(bookfd);
		}
		printf("Credit remaining: %d units\n", USAGEMAX - ausage);
	}
	exit(0);
}

/*
 * Determine the name of the tbs command
 */
enum command
findname()
{
	register char	*s, *sp;
	register int 	i;

	s = sp = argv[0];
	while(*sp++)
		;
	while(*--sp != '/' && sp >= s)
		;
	sp++;
	for(i = 0; i < NUMCOMS; i++)
		if(strcmp(comlist[i].com, sp) == 0)
			return(comlist[i].comval);
	error("Not a valid terminal booking system command");
}

/*
 * Issue message to explain why book/free failed
 */
explain()
{
	flushprint();
	if(hitusage)
		printf("Booking limit reached\n");
	else if(hitrate)
		printf("Booking density limit reached\n");
	else if(hitbooked)
		printf("Already booked for period\n");
	else if(hitunusable && sterm && not stermrange)
		printf("Terminal unusable\n");
	else if(hitunbookable && stime && not stimerange)
		printf("Unbookable period\n");
	else if(nextbest)
		printf("Unable to book for period specified\n");
}



/************************************************************************************************************
 *
 * Routines to transform arguments given to program to search parameters
 *
 ************************************************************************************************************/

int		argerr, intval, arg, timeval, lasterr;

/*
 * Extract arguments from argv[], checking for correctness and setting
 * up global search parameters 
 */
getargs()
{
	register enum	symbol val, sval;

	argerr = false;
	val = nextarg();
	while(val != Eof)
	{
		switch(val)
		{
		case Sunday:
		case Monday:
		case Tuesday:
		case Wednesday:
		case Thursday:
		case Friday:
		case Saturday:
			if(sday++)
				reperror("day");
			day = (int)val - (int)Sunday;
			break;
		case Today:
			if(sday++)
				reperror("day");
			day = tim->dayofwk;
			break;
		case Tomorrow:
			if(sday++)
				reperror("day");
			day = (tim->dayofwk + 1) % 7;
			break;
		case Here:
			if(sterm++)
				reperror("tty");
			termlo = findterm();
			break;
		case User:
			if(suser++)
				reperror("user");
			if((val = nextarg()) != Eof)
				user = finduid(argv[arg]);
			else
			{
				argerror("login name expected");
				continue;
			}
			break;
		case Logname:
			if(suser++)
				reperror("user");
			user = finduid(argv[arg]);
			break;
		case Video:
		case Hardcopy:
			if(sttype++)
				reperror("type");
			ttype = (int)val;
			break;
		case For:
			if(sforval++)
				reperror("for");
			if((val = nextarg()) == Int)
				forval = intval;
			else
			{
				argerror("integer expected");
				continue;
			}
			break;
		case Tty:
			if(sterm++)
				reperror("tty");
			if((termlo = ttytoterm(argv[arg])) < 0)
				argerror("unknown tty name");
			if((val = nextarg()) != To)
				continue;
			stermrange = true;
			if((val = nextarg()) == Tty)
			{
				if((termhi = ttytoterm(argv[arg])) < 0)
					argerror("unknown tty name");
			}
			else
			{
				argerror("tty name expected");
				continue;
			}
			if(termhi < termlo)
			{
				int tmp;

				tmp = termhi;
				termhi = termlo;
				termlo = tmp;
			}
			break;
		case Time:
		case Int:
		case Now:
			if(stime++)
				reperror("time");
			switch(val)
			{
			case Int:
				timelo = intval * PERIODSINHOUR;
				break;
			case Time:
				timelo = timeval;
				break;
			case Now:
				timelo = currentperiod;
				break;
			}
			sval = val;
			switch(val = nextarg())
			{
			case Am:
#ifdef AMPM
				if(inrange(timelo, 12 * PERIODSINHOUR, 12 * PERIODSINHOUR + 1))
					/* 12:00am, 12:30am ==> 0:00, 0:30 */
					timelo -= 12 * PERIODSINHOUR;
#endif AMPM
				val = nextarg();
				break;
			case Pm:
				if(timelo < 12 * PERIODSINHOUR)
					/* 0:00pm .. 11:30pm ==> 12:00 .. 23:30 */
					timelo += 12 * PERIODSINHOUR;
				val = nextarg();
				break;
			default:
				if(timelo < 12 * PERIODSINHOUR && sval != Now)
					argerror("must specify \"am\" or \"pm\" after a time");
				break;
			}
			if(timelo == 24 * PERIODSINHOUR)
				timelo = 0;
			if(not inrange(timelo, 0 , 24 * PERIODSINHOUR - 1))
				argerror("time out of range");
			if(val != To)
				continue;
			stimerange = true;
			switch(val = nextarg())
			{
			case Int:
				timehi = intval * PERIODSINHOUR;
				break;
			case Now:
				timehi = currentperiod;
				break;
			case Time:
				timehi = timeval;
				break;
			default:
				argerror("time expected");
				continue;
			}
			sval = val;
			switch(val = nextarg())
			{
			case Am:
#ifdef AMPM
				if(inrange(timehi, 12 * PERIODSINHOUR, 12 * PERIODSINHOUR + 1))
					timehi -= 12 * PERIODSINHOUR;
#endif AMPM
				break;
			case Pm:
				if(timehi < 12 * PERIODSINHOUR)
					timehi += 12 * PERIODSINHOUR;
				break;
			default:
				if(timehi < 12 * PERIODSINHOUR && sval != Now)
					argerror("must specify \"am\" or \"pm\" after a time");
				break;
			}
			if(timehi == 0)
				timehi = 24 * PERIODSINHOUR;
			if(not inrange(timehi, 1, 24 * PERIODSINHOUR))
				argerror("time out of range");
			if(timehi <= timelo)
				argerror("second time must be greater than first");
			if(val != Am && val != Pm)
				continue;
			break;
		case Am:
		case Pm:
			argerror("must be preceded by an integer");
			break;
		case To:
			argerror("must be preceded by a time");
			break;
		case String:
			argerror("unidentified string");
			break;
		default:
			snark("getargs");
		}
		val = nextarg();
	}
	return(argerr);
}

/*
 * Get the type and value (if applicable) of the next argument
 */
enum symbol
nextarg()
{
	int		hour, min;
	enum symbol	srchret;
	enum symbol	search(), type();
	char		m;
	static char	ampm;
	static bool	ampmkludge = false;

	if(ampmkludge)
	{
		ampmkludge = false;
		return(ampm == 'a' ? Am : Pm);
	}
	if(++arg >= argc)
		return(Eof);
	switch(type(argv[arg]))
	{
	case Int:
		intval = atoi(argv[arg]);
		return(Int);
	case Long:
		return(Logname);
	case String:
		if((srchret = search(argv[arg])) != String)
			return(srchret);
		min = ampm = m = 0;
		if((sscanf(argv[arg], "%d:%d%c%c", &hour, &min, &ampm, &m) >= 2
		|| sscanf(argv[arg], "%d%c%c", &hour, &ampm, &m) >= 1)
		&& (((ampm == 'a' || ampm == 'p') && m == 'm') || ampm == 0))
		{
			timeval = hour * PERIODSINHOUR + min / MINSINPERIOD;
			ampmkludge = (ampm != 0);
			return(Time);
		}
		return(String);
	default:
		snark("nextarg");
	}
}

snark(p)
register char *p;
{
	flushprint();
	fputs("SNARK: ", stdout);
	fflush(stdout);
	perror(p);
	exit(-1);
}

/*
 * Get the "type" of a given string
 */
enum symbol
type(p)
register char *p;
{
	register char	*tp;

	tp = p;
	while(*p)
		if(*p < '0' || *p++ > '9')
			return(String);
	return(p - tp > 4 ? Long : Int );
}

/*
 * Search keylist for a recognisable string
 */
enum symbol
search(p)
char *p;
{
	register int	i, j, k;

	lowercase(p);
	i = 0;
	j = NUMKEYS;
	while(i + 1 != j)	/* keylist[i].key <= p < keylist[j].key */
	{
		k = (i + j) / 2;
		if(cmp3(keylist[k].key, p) <= 0)
			i = k;
		else
			j = k;
	}
	if(cmp3(keylist[i].key, p) == 0)
		return(keylist[i].keyval);
	else
		return(String);
}

/*
 * Convert a string to lower case 
 */
lowercase(p)
register char *p;
{
	while(*p)
	{
		if(*p >= 'A' && *p <= 'Z')
			*p += 'a' - 'A';
		p++;
	}
}

/*
 * Compare first 3 char of string, returns number < 0, == 0 or > 0
 */
cmp3(a, b)
register char *a, *b;
{
	register int	i = 3;

	while(*a++ == *b++ && a[-1] && --i)
		;
	return(*--a - *--b);
}

reperror(p)
register char *p;
{
	if(arg == lasterr)
		/* only one error per argument! */
		return;
	lasterr = arg;
	errline();
	printf("repeated %s argument\n", p);
	argerr = true;
}

/*
 * Print pointer to error on the command line.
 * This assumes a normal prompt and one space between each parameter
 */
errline()
{
	register int	i, j, k;

	fputs("--", stdout);
	for(i = 0; i < arg; i++)
	{
		j = size(argv[i]);
		for(k = 0; k <= j; k++)
			fputs("-", stdout);
	}
	printf("^-- ");
}

/*
 * Size of a string
 */
size(s)
register char *s;
{
	register char	*ts;

	ts = s;
	while(*s++)
		;
	return(s - ts - 1);
}

/*
 * Get passwd file entry given login name
 */
finduid(lname)
register char *lname;
{
	pe.pw_strings[LNAME] = lname;
	if(getpwuid(&pe, sbuf, sizeof sbuf) == PWERROR)
		argerror("unknown login name");
	return(pe.pw_uid);
}

argerror(p)
register char *p;
{
	if(arg == lasterr)
		/* only one error per argument! */
		return;
	lasterr = arg;
	errline();
	printf("%s\n", p);
	argerr = true;
}

/************************************************************************************************************
 *
 * Initialization routines
 *
 ************************************************************************************************************/

init()
{
	long		mtime;
	bool		modhead = false;

	ulimit(2, TBULIMIT);
	uid = getuid();
	systime = time((long *)0);
	tim = localtime(systime);
	systime -= timezone;
	if(tim->tm_isdst)
		systime += 3600;	/* one hour */
	currentperiod = systime / SECSINPERIOD % NUMPERIODS;
	setbuf(stdout, obuf);		/* buffer output */

	lockbookfile(WRITE);
	rewrite = true;		/* causes getperiod to rewrite periods accesed */

	/* read in the header */
	if(lseek(bookfd, longoffset(b_header), 0) == SYSERROR)
		snark("init-lseek");
	if(read(bookfd, &head, sizeof(header)) != sizeof(header))
		snark("init-read");

	/*
	 * Update booking file if
	 *	1/ "/etc/ttys" file has been modified
	 *	2/ costfile has been modified
	 *	3/ More than one booking period has past
	 */
	mtime = modtime(etcttys);
	if(mtime != head.utimes.u_etcttys)
	{
		setupterminfo();
		head.utimes.u_etcttys = mtime;
		modhead = true;
	}
	mtime = modtime(costfile);
	if(mtime != head.utimes.u_costfile)
	{
		setupcosterminfo();
		head.utimes.u_costfile = mtime;
		modhead = true;
	}
	mtime = systime / SECSINPERIOD;
	if(not is(Cunbook) && mtime > head.utimes.u_bookfile)
	{
		update();
		head.utimes.u_bookfile = mtime;
		modhead = true;
	}
	flushblocks();		/* write out all unwritten modified periods */

	if(modhead)
	{
		sync();		/* ensure update times are written last */

		/* write out updated header */
		if(lseek(bookfd, longoffset(b_header), 0) == SYSERROR)
			snark("init-lseek1");
		if(write(bookfd, &head, sizeof(header)) != sizeof(header))
			snark("init-write");
	}
	unlock(bookfd);
	rewrite = false;
}

/*
 * Modification time of a file
 */
long modtime(fname)
register char *fname;
{
	struct stat	buf;

	if(stat(fname, &buf) == SYSERROR)
		snark("modtime");
	return(buf.st_mtime);
}

/*
 * Update terminal information from "etc/ttys"
 * Invalid lines are ignored.
 */
setupterminfo()
{
	char			type, group, sign[2], tname[40];
	int			num;
	register FILE 		*ttys;
	register terminfo 	*tp;
	char 			iobuf[BUFSIZ];
	char 			line[128];

	if(not (ttys = fopen(etcttys, "r")))
		snark("setuptinfo");
	setbuf(ttys, iobuf);

	for(tp = &head.tinfo[0]; tp < &head.tinfo[NUMTERMS]; tp++)
		tp->t_name[0] = tp->t_type = tp->t_usable = 0;

	while(not feof(ttys))
		if(fgets(line, sizeof line, ttys) == line
		&& sscanf(line, "%*c%c%c%s%*[\t ]%1[+-]%d", &type, &group, tname, sign, &num) == 5
		&& sign[0] != '\0'
		&& inrange(num, TERMMIN, TERMMAX))
		{
			num -= TERMMIN;
			tp = &head.tinfo[num];
			strncpy(tp->t_name, tname, sizeof tp->t_name);
			switch(type)
			{
#ifdef EECF
		case 'd':	/* decwriter */
				tp->t_type = (int)Hardcopy;
				break;
		case 'S':	/* Serge video terminal */
				tp->t_type = (int)Video;
				break;
		case 'Z':	/* Express video terminal */
				tp->t_type = (int)Video | EXPRESS;
				break;
#else EECF
		case '4':	/* VC 404 */
		case '5':
		case '6':
		case '7':	/* T1061 */
				tp->t_type = (int)Video;
				break;

		case 'N':	/* NCR 260 */
		case '3':	/* TTY 43 */
		case 'l':	/* LA 36 */
				tp->t_type = (int)Hardcopy;
				break;
#endif EECF
		default:
				tp->t_type = (int)Hardcopy | (int)Video;
				break;
			}
			tp->t_usable = (sign[0] == '+');
		}
	fclose(ttys);
}

/*
 * Update period cost information from costfile
 * A negative cost denotes an unbookable period.
 * Invalid lines are ignored.
 */
setupcosterminfo()
{
	int			timelo, timehi, cost, day, seclo, sechi;
	register int		i;
	register FILE		*costf;
	register periodtype	*tp;
	periodtype		*getperiod();
	char			iobuf[BUFSIZ], daystring[10], line[80];

	if(not (costf = fopen(costfile, "r")))
		snark("setupcosterminfo");
	setbuf(costf, iobuf);

	/* Initialise periods to default cost */
	for(i = 0; i < TOTALPERIODS; i++)
	{
		tp = getperiod(i);
		tp->timecost = DEFCOST;
	}

	while(not feof(costf))
		if(fgets(line, sizeof line, costf) == line
		&& sscanf(line, "%s %d:%d %d:%d %d", daystring, &timelo, &seclo, &timehi, &sechi, &cost) == 6
		&& inrange(timelo, 0, 24)
		&& inrange(seclo, 0, 59)
		&& inrange(timehi, 0, 24)
		&& inrange(sechi, 0, 59)
		&& inrange(cost, -1, COSTMAX)
		&& inrange((day = (int)search(daystring) - (int)Sunday), 0, 6))
		{
			timehi = timehi * PERIODSINHOUR + sechi / SECSINPERIOD;
			timelo = timelo * PERIODSINHOUR + seclo / SECSINPERIOD;
			if(timehi == 0)
				timehi = 24 * PERIODSINHOUR;
			for(i = timelo; i < timehi; i++)
			{
				tp = getperiod(day * NUMPERIODS + i);
				tp->timecost = cost;
			}
		}
	fclose(costf);
}

/*
 * Update user usage figures and
 * Clear out all periods that have passed (time-wise), except for class bookings
 */
update()
{
	register periodtype	*pp;
	register termtype	*termp, *ttermp;
	ushort			credit;
	int			timec, period, nperiods;

	if(head.utimes.u_bookfile == 0)	/* new file */
		return;
	/* adjust users usage figures */
	credit = systime / SECSINPERIOD - head.utimes.u_bookfile;
	if(credit > USAGEMAX)
		credit = USAGEMAX;
	if(credit)
	{
		register usagetype *up;
		userusage u;

		/* read in user usage section of bookfile */
		if(lseek(bookfd, longoffset(b_usage), 0) == SYSERROR)
			snark("update-lseek");
		if(read(bookfd, &u, sizeof(userusage)) != sizeof(userusage))
			snark("update-read");

		for(up = &u.val[0]; up < &u.val[NUSERS]; up++)
			if(credit >= *up)
				*up = 0;
			else
				*up -= credit;

		/* write back updated usage figures */
		if(lseek(bookfd, longoffset(b_usage), 0) == SYSERROR)
			snark("update-lseek1");
		if(write(bookfd, &u, sizeof(userusage)) != sizeof(userusage))
			snark("update-write");

		/* clear out old bookings */
		nperiods = credit;
		period = (head.utimes.u_bookfile + 4 * NUMPERIODS) % TOTALPERIODS; 
		/* + 4 since Jan 01 00:00:00 1970 was a Thursday */
	
		for(timec = 0; timec < nperiods; timec++)
		{
			pp = getperiod(period++);
			for(termp = &pp->terms[0], ttermp = &pp->terms[NUMTERMS]; termp < ttermp; termp++)
					if(not termp->bits.t_class)
						termp->whole = 0;
		}
	}
}

/*
 * Lock the booking file for reading or writing.
 * If not already open do so.
 * If booking file doesn't exist a new one is created.
 */
lockbookfile(mode)
int mode;
{
	static bool		bookopen = false;
	char			c = 0;

	if(not bookopen)
	{
		if((bookfd = open(bookingfile, READWRITE)) == SYSERROR)
		{
			/* try to create a new booking file */
			if(mknod(bookingfile, S_IFLOK|0644, 0) == SYSERROR
			|| (bookfd = open(bookingfile, READWRITE)) == SYSERROR
			|| writelock(bookfd) == SYSERROR
			|| lseek(bookfd, (long)(sizeof(bookfile) - 1), 0) == SYSERROR
			|| write(bookfd, &c, 1) != 1
			|| close(bookfd) == SYSERROR)
				snark("lockbookfile-creat");
			if((bookfd = open(bookingfile, READWRITE)) == SYSERROR)
				snark("lockbookfile-open");
		}
		bookopen = true;
	}
	if(mode == READ)
	{
		if(readlock(bookfd) == SYSERROR)
			snark("lockbookfile-readlock");
	} 
	else
		if(writelock(bookfd) == SYSERROR)
			snark("lockbookfile-writelock");
}

/*
 * Get passwd entry given uid
 */
findlog(uid)
register int uid;
{
	if(pe.pw_uid != uid)
	{
		pe.pw_uid = uid;
		if(getpwlog(&pe, sbuf, sizeof sbuf) == PWERROR)
		{
			/*
			 * should only happen if account is deleted after a
			 * booking has been made.
			 */
			pe.pw_strings[LNAME] = "???????";
			pe.pw_flags = 0;
			return(-1);
		}
	}
	return(pe.pw_uid);
}

/*
 * Find current term type
 */
findttype()
{
	return(head.tinfo[findterm()].t_type);
}

/*
 * Find current terminal number, if not a term, returns term 0
 */
findterm()
{
	static bool		used = false;
	static int		term = 0;
	char			*ttyname();
	register char		*cp;

	if(not used)
	{
		used = true;
		cp = ttyname(ERROROUT);
		if(strncmp(cp, "/dev/", 5) == 0)
			cp += 5;
		if((term = ttytoterm(cp)) < 0)
			term = 0;
	}
	return(term);
}


/*
 * Convert ttyname to term number
 */
ttytoterm(s)
char *s;
{
	register terminfo	*tp;
	register int		term;

	tp = &head.tinfo[0];
	term = 0;
	while(strncmp(s, tp->t_name, sizeof tp->t_name) && tp < &head.tinfo[NUMTERMS])
		tp++, term++;
	if(term == NUMTERMS)
		return(-1);
	return(term);
}

/************************************************************************************************************
 *
 * Routines to set default search parameters
 *
 ************************************************************************************************************/

/*
DEFAULTS FOR TBS COMMANDS
-------------------------------------------------------------------------------
NORMAL USER	book		unbook		list		free
-------------------------------------------------------------------------------
day		current		current		current		current
timelo		<- - - -     (day == today ? current : 0am)	- - - ->
timehi		<- - - -     (day == today ? timelo : 12pm)	- - - ->
user		current		current		current		current
ttype		current		current		all types	all types
forval		1		1		large		10
termlo		current		current		current		current
termhi		current		current		current		current
-------------------------------------------------------------------------------
CLASS		book		unbook		list		free
-------------------------------------------------------------------------------
day		*		*		Sunday		Sunday
timelo		*		*		0am		0am
timehi		timelo		timelo		12pm		12pm
user		current		current		current		current
ttype		all types	all types	all types	all types
forval		no. requested	large		large		10
termlo		*		minterm		minterm		minterm
termhi		termlo		maxterm		maxterm		maxterm
-------------------------------------------------------------------------------
SUPER USER	book		unbook		list		free
-------------------------------------------------------------------------------
day		/\		* or Sunday	Sunday		Sunday
timelo		| as for	* or 0am	0am		0am
timehi		| normal	timelo or 12pm	12pm		12pm
user		*		* or All user	All users	All users
ttype		| user or	all types	all types	all types
forval		| class		large		large		10
termlo		|		minterm		minterm		minterm
termhi		\/		maxterm		maxterm		maxterm
-------------------------------------------------------------------------------
NOTES:	* = mandatory; for su unbooking either day and time or user is mandatory
-------------------------------------------------------------------------------
*/

/*
 * Set up default parameters for super users.
 */
susetdef()
{
	switch(thiscommand)
	{
	case Book:
	case Evict:
		if(not suser)
			error("Super users must specify a user");
		else
		{
			setdef();	/* set defaults for the appropriate normal or class user */
			return;
		}
	case Unbook:
		if(not (sday && stime || suser))
			error("Super users must specify time and day or a user for unbooking");
		break;
	case Cunbook:
		if(not (sday && stime))
			error("Post-crash unbookings must have day and time specified");
		break;
	}
	if(not suser)
		if((user = findlog(uid)) < 0)
			snark("susetdef");
	if(not sday)
		day = 0;	/* Sunday */
	if(not stime)
		timelo = 0;	/* 0 am */
	if(not stimerange)
		if(stime)
			timehi = timelo;
		else
			timehi = 24 * PERIODSINHOUR - 1;	/* 12 pm */
	else
		timehi -= 1;
	if(not sttype)
		ttype = (int)Hardcopy | (int)Video;
	if(not forval)
		if(is(Free))
			forval = DEFFREE;
		else
			forval = BIG;
	if(not sterm)
		termlo = 0;
	if(not stermrange)
		if(sterm)
			termhi = termlo;
}

error(s)
char *s;
{
	printf("%s\n", s);
	exit(-1);
}

/*
 * Set unset parameters to default values, defaults are usually current day, time,
 * term and term type.
 */
setdef()
{
	if(not suser)
		if((user = findlog(uid)) < 0)
			snark("setdef");
	if(class)
	{
		if(is(Book) && (not sday || not stime || not sterm))
			error("Class bookings must have day, time and terminals specified");
		if(is(Unbook) && (not sday || not stime))
			error("Class unbookings must have day and time specified");
	}
	if(not sday)
		if(class && not is(Evict))
			day = 0;	/* Sunday */
		else
			day = tim->dayofwk;
	if(not stime)
		if(class && not is(Evict) || day != tim->dayofwk)
			timelo = 0;	/* 0 am */
		else
			timelo = currentperiod;
	if(not stimerange)
		if(class && not is(Evict) && (is(List) || is(Free))
		|| not class && day != tim->dayofwk && not stime)
			timehi = 24 * PERIODSINHOUR - 1;	/* 12 pm */
		else
			timehi = timelo;
	else
		timehi -= 1;
	if(not sttype)
		if(not class && (is(Book) || is(Unbook)))
			ttype = findttype();
		else
			ttype = (int)Hardcopy | (int)Video;
	if(not sterm)
		if(class)
			termlo = 0;
		else
			termlo = findterm();
	if(not stermrange)
		if(not class || sterm)
			termhi = termlo;
	if(not sforval)
		if(class)
			if(is(Free))
				forval = DEFFREE;
			else if(is(Book))
				forval = (timehi - timelo + 1) * (termhi - termlo + 1);
			else
				forval = BIG;
		else
			if(is(Free))
				forval = DEFFREE;
			else if(is(List))
				forval = BIG;
			else /* Book, Unbook, Evict */
				forval = 1;
}


/************************************************************************************************************
 *
 * Search routines
 *
 ************************************************************************************************************/

/*
 * Find free periods starting from (day, time, term) and call func on each
 * occurance. When func returns non-zero stop.
 * Trys to find terminal type requested.
 */
findfree(day, time, term, func)
int day, time, term, (*func)();
{
	register periodtype	*timep;
	register termtype	*termp, *ttermp;
	register int		termc, timec, sfterm;
	bool			blockbook;
	int			period, found = 0, iperiod;

	blockbook = (is(Book) && not class && sforval && forval <= RATEPERIOD);
	iperiod = period = timetoperiod(day, time);
	for(timec = 0; timec < TOTALPERIODS; timec++) 	/* for each time period */
	{
		timep = getperiod(period++);
		if(timep->timecost >= 0) 	/* bookable time period */
		{
			sfterm = -1;
			termp = &timep->terms[term];
			ttermp = &timep->terms[NUMTERMS];
			for(termc = 0; termc < NUMTERMS; termc++)	/* for each terminal */
			{
				if(not termp->bits.t_booked)
				{
					fterm = (term + termc) % NUMTERMS;
					fday = (day + (time + timec) / NUMPERIODS) % 7;
					ftime = (time + timec) % NUMPERIODS;
					if(head.tinfo[fterm].t_usable
					&& ((head.tinfo[fterm].t_type & EXPRESS) == 0 || class))	/* a usable terminal */
					{
						if(sterm || blockbook || (head.tinfo[fterm].t_type & ttype))
						{
							found++;
							if(blockbook)
								fterm = term = findblock(period - 1, fterm, forval);
							if((*func)())
								return(found);
							sfterm = -1;
							if(not class)		
								/* prevent multiple bookings */
								break;
						} 
						else
							sfterm = fterm;
					} 
					else
						hitunusable = (hitunusable || rangewanted());
				}
				if(++termp >= ttermp)
					termp = &timep->terms[0];
			}
			if(sfterm >= 0)	/* found but not type reqested */
			{
				found++;
				fterm = sfterm;
				if((*func)())
					return(found);
			}
		} 
		else
			hitunbookable = (hitunbookable || (period - 1) == iperiod);
	}
	return(found);
}

/*
 * Finds the largest block up to "size" periods long, starting from a given
 * period.
 */
findblock(startperiod, term, size)
int startperiod, term, size;
{
	register periodtype	*pp;
	register termtype	*tp;
	register int		termc, fterm, sfterm;
	register bool		found;
	int			lfterm, timec;
	bool			map[NUMTERMS];

	for(termc = 0; termc < NUMTERMS; termc++)
		map[termc] = head.tinfo[termc].t_usable;
	lfterm = fterm = term;
	for(timec = 0; timec < size; timec++)
	{
		pp = getperiod(startperiod++);
		if(pp->timecost >= 0)
		{
			sfterm = -1;
			found = false;
			tp = &pp->terms[0];
			for(termc = 0; termc < NUMTERMS; termc++)
			{
				if((map[termc] = map[termc]
				& (tp->whole == 0
				&& (not sttype || (head.tinfo[termc].t_type & ttype)))) 
				&& not found)
					if(not sttype || (head.tinfo[termc].t_type & ttype))
					{
						found = true;
						fterm = termc;
					}
					else
						sfterm = termc;
				tp++;
			}
			if(not found)
				if(sfterm < 0)
				{
					fterm = lfterm;
					break;
				}
				else
					fterm = sfterm;
			lfterm = fterm;
		}
	}
	if(map[term])
		return(term);
	return(fterm);
}

/*
 * Find periods booked by user starting from (day, time, term) and call
 * func on each occurance. When func returns non-zero stop.
 */
findused(day, time, term, user, func)
int day, time, term, user, (*func)();
{
	register periodtype	*timep;
	register termtype	*termp, *ttermp;
	register int		termc, timec, period, found = 0;

	period = timetoperiods(day, time);
	for(timec = 0; timec < TOTALPERIODS; timec++)
	{
		timep = getperiod(period++);
		if(timep->timecost >= 0)
		{
			termp = &timep->terms[term];
			ttermp = &timep->terms[NUMTERMS];
			for(termc = 0; termc < NUMTERMS; termc++)
			{
				if(termp->bits.t_uid == user && termp->bits.t_booked)
				{
					found++;
					fday = (day + (time + timec) / NUMPERIODS) % 7;
					ftime = (time + timec) % NUMPERIODS;
					fterm = (term + termc) % NUMTERMS;
					if((*func)())
						return(found);
					if(not class)
						break;
				}
				if(++termp >= ttermp)
					termp = &timep->terms[0];
			}
		}
	}
	return(found);
}

/*
 * Super user findused:
 * Find booked periods starting from (day, time, term) and call
 * func on each occurance. When func returns non-zero stop.
 */
sufindused(day, time, term, func)
int day, time, term, (*func)();
{
	register periodtype	*timep;
	register termtype	*termp, *ttermp;
	register int		termc, timec, period, found = 0;

	period = timetoperiods(day, time);
	for(timec = 0; timec < TOTALPERIODS; timec++)
	{
		timep = getperiod(period++);
		if(timep->timecost >= 0)
		{
			termp = &timep->terms[term];
			ttermp = &timep->terms[NUMTERMS];
			for(termc = 0; termc < NUMTERMS; termc++)
			{
				if(termp->bits.t_booked)
				{
					found++;
					fday = (day + (time + timec) / NUMPERIODS) % 7;
					ftime = (time + timec) % NUMPERIODS;
					fterm = (term + termc) % NUMTERMS;
					if((*func)(termp->bits.t_uid))
						return(found);
				}
				if(++termp >= ttermp)
					termp = &timep->terms[0];
			}
		}
	}
	return(found);
}


/************************************************************************************************************
 *
 * Action routines called by search routines
 *
 ************************************************************************************************************/

/* vars used by print and flushprint */

int	luid, lday, ltime, lterm, ltime2, llterm;
bool	lnext, lclass, merge = false;
#ifdef AMPM
bool	lspm2;
#endif AMPM

/*
 * Book for normal users
 */
book()
{
	if(limit() < 0)
		/* a limit reached */
		return(0);
	if(rangewanted())
	{
		checkpw();
		if(dobook(user, pe.pw_tbrate))
			return(0);
		print("Booked");
		return(--forval == 0);
	}
	nextbest = true;
	flushprint();
	print("Next Best");
	return(-1);
}

/*
 * Book for classes
 */
cbook()
{
	if(rangewanted())
	{
		checkpw();
		if(dobook(user, pe.pw_tbrate))
			return(0);
		print("Booked");
		return(--forval == 0);
	}
	if(iforval != forval)
		return(not prangewanted());
	nextbest = true;
	print("Next Best");
	return(-1);
}

/*
 * Super user list, needs to look up passwd file entry for each user.
 */
sulist(user)
int user;
{
	if(rangewanted())
	{
		findlog(user);
		print("Booked");
		return(--forval == 0);
	}
	return(not prangewanted());
}

/*
 * Super user unbook, needs to look up passwd file entry for each user.
 */
suunbook(user)
int user;
{
	if(rangewanted())
	{
		findlog(user);
		if(class)
			return(0);	/* SU's must explicitly unbook classes */
		if(dounbook(user, pe.pw_tbrate))
			return(0);
		print("Unbooked");
		return(--forval == 0);
	}
	return(not prangewanted());
}

/*
 * Print free periods
 */
pfree()
{
	if(rangewanted() && (class || (su && not suser) || limit() >= 0))
		return((forval -= print("Free")) == 0);
	return(not prangewanted());
}

/*
 * Print periods booked by a particular user.
 */
list()
{
	if(rangewanted())
	{
		print("Booked");
		return(--forval == 0);
	}
	return(not prangewanted());
}

/*
 * Unbook for normal users and classes
 */
unbook()
{
	if(not (class || su) && fday == tim->dayofwk && ftime == currentperiod)
		hitcurrent = true;		/* can't unbook current time */
	else if(rangewanted())
	{
		checkpw();
		if(dounbook(user, pe.pw_tbrate))
			return(0);
		print("Unbooked");
		return(--forval == 0);
	}
	if(class && forval != iforval)
		return(0);
	return(not prangewanted());
}

/*
 * Form a list of terminals to evict.
 */
mkevictlist()
{
	if(rangewanted() && ftime == currentperiod)
	{
		checkpw();
		*elistptr++ = head.tinfo[fterm].t_name;
		return(--forval == 0);
	}
	return(ftime != currentperiod);
}

/************************************************************************************************************
 *
 * Routines called by action routines
 *
 ************************************************************************************************************/

/*
 * Returns true if period found is within the range of day, time, term, term type specified.
 */
rangewanted()
{
	return(	(not sday || fday == day)
		&& (not stime || inrange(ftime, timelo, timehi))
		&& (not sterm || inrange(fterm, termlo, termhi))
		&& (not sttype || (head.tinfo[fterm].t_type & ttype))
	);
}

/*
 * Returns true if period is in the correct day and time range only.
 */
prangewanted()
{
	return(fday == day && (not stime || inrange(ftime, timelo, timehi)) || not sday);
}

/*
 * Find if booking at (fday, ftime) is permissible, and return the cost if so.
 * There are three constraints:
 *	1/ One booking per period
 *	2/ Max usage limit
 *	3/ Max rate of usage limit
 */
limit()
{
	register periodtype	*pp;
	register termtype	*termp;
	register int		cost;

	pp = getperiod(timetoperiod(fday, ftime));
	for(termp = &pp->terms[0]; termp < &pp->terms[NUMTERMS]; termp++)
		if(termp->bits.t_uid == user && termp->bits.t_booked)
			break;
	if(termp != &pp->terms[NUMTERMS])
	{
		/* already booked for current period (at some other term)*/
		hitbooked += rangewanted();
		return(-1);
	} 
	cost = pp->timecost * pe.pw_tbrate;
	if(not gotusage)
	{
		lockbookfile(READ);
		getusage(user);
		unlock(bookfd);
		gotusage = true;
	}
	if(ausage + cost > USAGEMAX)
	{
		/* exceeds max usage */
		hitusage += rangewanted();
		return(-1);
	} 
	if(findcost(RATEPERIOD) + cost > unsign(pe.pw_tblim))
	{
		/* exceeds max rate of booking */
		hitrate += rangewanted();
		return(-1);
	} 
	return(cost);
}

/*
 * Used to enforce consecutive booking limit.
 * Finds maximum sum of booking costs over a "num" hour range,
 * checking num-1 hours either side of fday, ftime.
 */
findcost(num)
register int num;
{
	register periodtype	*timep;
	register termtype	*termp;
	register int		timec, found, timec2;
	int			period, snum, maxfound = 0;
	termtype		*ttermp;
	int			cost[RATEPERIOD * 2 + 1];

	num--;
	period = (timetoperiod(fday, ftime) - num + TOTALPERIODS ) % TOTALPERIODS;
	snum = num * 2 + 1;
	num++;
	for(timec = 0; timec < snum; timec++)
	{
		timep = getperiod(period++);
		cost[timec] = 0;
		if(timep->timecost >= 0) 
			for(termp = &timep->terms[0], ttermp = &timep->terms[NUMTERMS]; termp < ttermp; termp++)
				if(termp->bits.t_uid == user && termp->bits.t_booked)
				{
					cost[timec] = timep->timecost * pe.pw_tbrate;
					break;
				}
	}
	/* now find max "num" hour period sum */
	for(timec = 0; timec < num; timec++)
	{
		found = 0;
		for(timec2 = 0; timec2 < num; timec2++)
			found += cost[timec + timec2];
		if(found > maxfound)
			maxfound = found;
	}
	return(maxfound);
}

struct sgttyb tty;

/*
 * Check passwd to allow booking/unbooking for other users
 */
checkpw()
{
	static bool		checked = false;
	char			line[80];
	extern			finish();

	if(not checked && not su && pe.pw_uid != uid && pe.pw_pword[0])
	{
		signal(2, finish);
		signal(3, finish);
		gtty(0, &tty);
		tty.sg_flags &= ~ECHO;
		stty(0, &tty);
		printf("%s Password: ", pe.pw_strings[LNAME]);
		fflush(stdout);
		if(pwcmp(crypt(gets(line), pe.pw_pword), pe.pw_pword))
		{
			printf("\nWrong Password");
			finish();
		}
		tty.sg_flags |= ECHO;
		stty(0, &tty);
		printf("\n");
		checked = true;
	}
}

finish()
{
	gtty(0, &tty);
	tty.sg_flags |= ECHO;
	stty(0, &tty);
	printf("\n");
	exit(-1);
}

/*
 * Compare encrypted passwds
 */
pwcmp(cp1, cp2)
register char *cp1, *cp2; 
{
	register int	i = CRYPTLEN;

	do
		if(*cp1++ != *cp2++)
			return(-1);
	while(--i);
	return(0);
}

/*
 * Carry out the booking, updating booking file.
 */
dobook(user, rate)
int user, rate;
{
	register periodtype	*pp;
	register int		newusage;
	register termtype	*tp;

	pp = getperiod(timetoperiod(fday, ftime));

	lockbookfile(WRITE);

	getterm();
	if(aterm.bits.t_booked)
	{
		/* shouldn't happen often - booked by another user */
		unlock(bookfd);
		return(-1);
	} 
	else
	{
		tp = &pp->terms[fterm];
		aterm.bits.t_uid = tp->bits.t_uid = user;
		if(class)
			aterm.bits.t_class = tp->bits.t_class = true;
		else
			aterm.bits.t_class = tp->bits.t_class = false;
		aterm.bits.t_booked = tp->bits.t_booked = true;
		putterm();

		if(not class)
		{
			getusage(user);
			if((newusage = ausage + pp->timecost * rate) > USAGEMAX)
				snark("dobook");
			ausage = newusage;
			putusage();
		}

		unlock(bookfd);
		return(0);
	}
}

/*
 * Carry out unbooking, updating booking file
 */
dounbook(user, rate)
int user, rate;
{
	register periodtype	*pp;
	register int		newusage;

	pp = getperiod(timetoperiod(fday, ftime));

	lockbookfile(WRITE);

	getterm();
	if(user == aterm.bits.t_uid && aterm.bits.t_booked)
	{
		aterm.whole = 0;
		putterm();

		if(not class)
		{
			getusage(user);
			if((newusage = ausage - pp->timecost * rate) < 0)
				ausage = 0;
			else
				ausage = newusage;
			putusage();
		}

		unlock(bookfd);
		return(0);
	} 
	else
	{
		/* Shouldn't happen often - unbooked by another user ! */
		unlock(bookfd);
		return(-1);
	}
}


/*
 * To extract a user usage from the booking file and move to core.
 */
getusage(user)
register int user;
{
	if(lseek(bookfd, longoffset(b_usage.val[user]), 0) == SYSERROR)
		snark("getusage-lseek");
	if(read(bookfd, &ausage, sizeof ausage) != sizeof ausage)
		snark("getusage-read");
}

/*
 * To be used directly after a "getusage" to write usage back into
 * the booking file.
 */
putusage()
{
	if(lseek(bookfd, - (long)(sizeof ausage), 1) == SYSERROR)
		snark("putusage-lseek");
	if(write(bookfd, &ausage, sizeof ausage) != sizeof ausage)
		snark("putusage-write");
}


/*
 * Similarly for the actual (day, time, term) word representing a booking
 */
getterm()
{
	if(lseek(bookfd, longoffset(b_period.days[fday].times[ftime].terms[fterm]), 0) == SYSERROR)
		snark("getterm-lseek");
	if(read(bookfd, &aterm, sizeof(termtype)) != sizeof(termtype))
		snark("getterm-read");
}

putterm()
{
	if(lseek(bookfd, - (long)(sizeof(termtype)), 1) == SYSERROR)
		snark("putterm-lseek");
	if(write(bookfd, &aterm, sizeof(termtype)) != sizeof(termtype))
		snark("putterm-write");
}

/*
 * Print out details of booking preceded by a string (like "Booked")
 * Consecutive hours at the same terminal are merged together.
 * Classes have terminal numbers merged instead.
 */
#ifdef AMPM
print(s)
char *s;
{
	bool		spm = false, spm2 = false, next, today;
	int		time2, time, alinedone = 0;
	static char	*lasts;

	if(lasts != s)
	{
		printf("%s:\n", s);
		lasts = s;
	}
	today = (fday == tim->dayofwk && ftime >= currentperiod);
	next = fday < tim->dayofwk || (fday == tim->dayofwk && ftime < currentperiod);
	time = ftime;
	time2 = ftime + 1;
	if(time >= 12 * PERIODSINHOUR)
	{
		spm = true;
		if(time > 12 * PERIODSINHOUR + 1)
			time -= 12 * PERIODSINHOUR;
	}
	else if(time <= 1)
		time += 12 * PERIODSINHOUR;
	if(time2 == 24 * PERIODSINHOUR)
		time2 = 0;
	if(time2 >= 12 * PERIODSINHOUR)
	{
		spm2 = true;
		if(time2 > 12 * PERIODSINHOUR + 1)
			time2 -= 12 * PERIODSINHOUR;
	}
	else if(time2 <= 1)
		time2 += 12 * PERIODSINHOUR;
	/* To merge or not to merge, that is the question.. */
	if(merge
	&& not(lday == fday
	   &&  luid == pe.pw_uid
	   &&  lnext == next
	   && ((ltime + 1 == ftime && not class) || (class && ltime == ftime))
	   && (lterm == fterm || (class && lterm + 1 == fterm) || is(Free)) 
	)  )
		flushprint();		/* sets merge to false */
	if(not merge)
	{
		if(su && not is(Free))
			printf(" user %-9s", pe.pw_strings[LNAME]);
		if(class)
			printf(" %s %s %2d:%02d%s to %2d:%02d%s %s",
				(next ? "Next" : today ? "Today  " : "This"),
				(today ? "" : daylist[fday]),
				time / PERIODSINHOUR,
				time % PERIODSINHOUR * MINSINPERIOD,
				(spm ? "pm" : "am"),
				time2 / PERIODSINHOUR,
				time2 % PERIODSINHOUR * MINSINPERIOD,
				(spm2 ? "pm" : "am"),
				head.tinfo[fterm].t_name
			);
		else
			printf(" %s %s %2d:%02d%s to", 
				(next ? "Next" : today ? "Today  " : "This"),
				(today ? "" : daylist[fday]),
				time / PERIODSINHOUR,
				time % PERIODSINHOUR * MINSINPERIOD,
				(spm ? "pm" : "am")
			);
		llterm = fterm;
		alinedone++;
	}
	lday = fday;
	ltime = ftime;
	lterm = fterm;
	luid = pe.pw_uid;
	lspm2 = spm2;
	ltime2 = time2;
	lnext = next;
	lclass = class;
	merge = true;
	return(alinedone);
}
#else AMPM
print(s)
char *s;
{
	bool		next, today;
	int		time2, time, alinedone = 0;
	static char	*lasts;

	if(lasts != s)
	{
		printf("%s:\n", s);
		lasts = s;
	}
	today = (fday == tim->dayofwk && ftime >= currentperiod);
	next = fday < tim->dayofwk || (fday == tim->dayofwk && ftime < currentperiod);
	time = ftime;
	time2 = ftime + 1;
	/* To merge or not to merge, that is the question.. */
	if(merge
	&& not(lday == fday
	   &&  luid == pe.pw_uid
	   &&  lnext == next
	   && ((ltime + 1 == ftime && not class) || (class && ltime == ftime))
	   && (lterm == fterm || (class && lterm + 1 == fterm) || is(Free)) 
	)  )
		flushprint();		/* sets merge to false */
	if(not merge)
	{
		if(su && not is(Free))
			printf(" user %-10s", pe.pw_strings[LNAME]);
		if(class)
			printf(" %s %s %2d:%02d to %2d:%02d %s",
				(next ? "Next" : today ? "Today  " : "This"),
				(today ? "" : daylist[fday]),
				time / PERIODSINHOUR,
				time % PERIODSINHOUR * MINSINPERIOD,
				time2 / PERIODSINHOUR,
				time2 % PERIODSINHOUR * MINSINPERIOD,
				head.tinfo[fterm].t_name
			);
		else
			printf(" %s %s %2d:%02d to", 
				(next ? "Next" : today ? "Today  " : "This"),
				(today ? "" : daylist[fday]),
				time / PERIODSINHOUR,
				time % PERIODSINHOUR * MINSINPERIOD
			);
			llterm = fterm;
		alinedone++;
	}
	lday = fday;
	ltime = ftime;
	lterm = fterm;
	luid = pe.pw_uid;
	ltime2 = time2;
	lnext = next;
	lclass = class;
	merge = true;
	return(alinedone);
}
#endif AMPM

/*
 * Called to print the second half of a merged booking details line.
 */
#ifdef AMPM
flushprint()
{
	if(merge)
		if(lclass)
			if(llterm != lterm)
				printf(" to %s\n", head.tinfo[lterm].t_name);
			else
				printf("\n");
		else if(is(Free))
			printf(" %2d:%02d%s\n", 
				ltime2 / PERIODSINHOUR,
				ltime2 % PERIODSINHOUR * MINSINPERIOD, 
				(lspm2 ? "pm" : "am")
			);
		else
			printf(" %2d:%02d%s %s\n", 
				ltime2 / PERIODSINHOUR,
				ltime2 % PERIODSINHOUR * MINSINPERIOD, 
				(lspm2 ? "pm" : "am"), 
				head.tinfo[lterm].t_name
			);
	fflush(stdout);			/* output a line */
	merge = false;
}
#else AMPM
flushprint()
{
	if(merge)
		if(lclass)
			if(llterm != lterm)
				printf(" to %s\n", head.tinfo[lterm].t_name);
			else
				printf("\n");
		else if(is(Free))
			printf(" %2d:%02d\n", 
				ltime2 / PERIODSINHOUR,
				ltime2 % PERIODSINHOUR * MINSINPERIOD 
			);
		else
			printf(" %2d:%02d %s\n", 
				ltime2 / PERIODSINHOUR,
				ltime2 % PERIODSINHOUR * MINSINPERIOD, 
				head.tinfo[lterm].t_name
			);
	fflush(stdout);			/* output a line */
	merge = false;
}
#endif AMPM

/************************************************************************************************************
 *
 * Period buffer handling routines
 *
 ************************************************************************************************************/

#define BLKSINBOOK	((sizeof(periods)) / 512 + 1)

/* the assumption is made that a period is less than 512 bytes long */

int	allocblock;		/* the next buffer to use */
bool	doinit = true;		/* whether or not initialization done */
char	buff[NUMBLKS][512];	/* the buffers */
char	incore[BLKSINBOOK];	/* maps block no. in file to block no. in core (if there) */
char	reverse[NUMBLKS];	/* maps block no. in core to block no. in file */

/*
 * Returns a pointer to the period structure requested from the booking file.
 * A pool of buffers is kept so that recently requested periods are quickly available.
 * If the flag "rewrite" is set, as buffers are reused the old contents are written
 * back into the booking file.
 * "flushblock" forces immediate rewriting of all used buffers.
 */
periodtype *
getperiod(period)
register int period;
{
	register int	sblock, eblock, i;

	if(doinit)
	{
		for(i = 0; i < BLKSINBOOK; i++)
			incore[i] = -1;
		for(i = 0; i < NUMBLKS; i++)
			reverse[i] = -1;
		doinit = false;
	}
	period %= TOTALPERIODS;
	sblock = (period * sizeof(periodtype)) / 512;
	eblock = ((period + 1) * sizeof(periodtype)) / 512;
	if(sblock == eblock) /* the period is totally contained in one block - no problem. */
	{
		if(incore[sblock] < 0)
			getblock(sblock, allocblock);
	} 
	else /* must organise things so that the two blocks are contiguous in core */
	     if(incore[sblock] >= 0 && incore[sblock] + 1 < NUMBLKS)
		getblock(eblock, incore[sblock] + 1);
	else if(incore[eblock] >= 0 && incore[eblock] - 1 >= 0)
	{
		getblock(sblock, incore[eblock] - 1);
		allocblock = (incore[eblock] + 1) % NUMBLKS;
	} 
	else if(incore[sblock] >= 0 || incore[eblock] >= 0 || allocblock + 1 == NUMBLKS)
	{
		getblock(eblock, 1);
		getblock(sblock, 0);
		allocblock = 2;
	} 
	else
	{
		getblock(sblock, allocblock);
		getblock(eblock, allocblock);
	}
	return((periodtype *)&buff[incore[sblock]][period * sizeof(periodtype) % 512]);
}

/*
 * After call the block required is found in buffer no. "dest"
 */
getblock(blockreq, dest)
int blockreq, dest;
{
	if(incore[blockreq] == dest)
	{
		return;
	}
	if(rewrite && reverse[dest] >= 0)
	{
		/* rewrite block about to be overwritten */
		if(lseek(bookfd, (long)(reverse[dest])<<9, 0) == SYSERROR)
			snark("getblock-lseek");
		if(write(bookfd, buff[dest], 512) != 512)
			snark("getblock-write");
	}
	if(incore[blockreq] >= 0)
	{
		/* block in core, just copy */
		register char *p1, *p2, *p3;

		p1 = buff[incore[blockreq]];
		p2 = buff[dest];
		p3 = p1 + 512;
		do
			*p2++ = *p1++;
		while(p1 < p3);
		reverse[incore[blockreq]] = -1;
	} 
	else
	{
		/* get block from file */
		if(not rewrite)
			lockbookfile(READ);
		if(lseek(bookfd, (long)(blockreq)<<9, 0) == SYSERROR)
			snark("getblock-lseek1");
		if(read(bookfd, buff[dest], 512) != 512)
			snark("getblock-read");
		if(not rewrite)
			unlock(bookfd);
	}
	if(reverse[dest] >= 0)
		incore[reverse[dest]] = -1;
	reverse[dest] = blockreq;
	incore[blockreq] = dest;
	allocblock = (dest + 1) % NUMBLKS;
}

/*
 * Rewrite all used buffers back into file.
 */
flushblocks()
{
	register int	i;

	if(doinit)
		return;			/* nothing to flush */
	for(i = 0; i < NUMBLKS; i++)
		if(reverse[i] >= 0)
		{
			if(lseek(bookfd, (long)(reverse[i])<<9, 0) == SYSERROR)
				snark("flushblocks-lseek");
			if(write(bookfd, buff[i], 512) != 512)
				snark("flushblocks-write");
			incore[reverse[i]] = -1;
			reverse[i] = -1;
		}
}

