/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)send.c	1.29 85/08/19";

/*
**	Send a message containing file transfer protocol.
**
**	SETUID ==> ROOT.
**
**	TODO:	add message logging.
**		add multi-part messages.
*/

char *	Usage	= "\"%s [-[A][C]] [-n<data-name>] \\\n\
	[<users>[@<destinations>] ...] [-] [[-f]file ...]\"";

#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO

#include	"global.h"

#include	"Passwd.h"
#include	"address.h"
#include	"command.h"
#include	"debug.h"
#include	"ftheader.h"
#include	"handlers.h"
#include	"header.h"
#include	"spool.h"
#include	"state.h"
#include	"sub_proto.h"

#include	<signal.h>


/*
**	Parameters set from arguments.
*/

bool	Ack;			/* Acknowledgement of delivery required */
bool	CRC;			/* Perform CRC on data */
char *	EnvString;		/* String for inclusion in header environment */
char *	DataName;		/* Name of file */
char *	Name;			/* Program invoked name */
bool	NoCall;			/* Don't auto-call for this message */
bool	NoOpt;			/* Don't optimise message delivery */
bool	NoRet;			/* Don't return message on error detection */
bool	NoFtph;			/* Don't add FTP header to message */
int	Traceflag;		/* Global tracing control */

/*
**	Destinations list.
*/

typedef struct Dlist *	Dlist_p;

typedef struct Dlist
{
	Dlist_p	d_next;		/* Next element at this level */
	Dlist_p*d_prev;		/* Back pointer for this level */
	Dlist_p	d_part;		/* Lower level complex address */
	char *	d_name;		/* Destination address if simple */
	char	d_sep;		/* Separator for this level */
}
	Dlist;

Dlist *	Destinations;		/* List of explicit destinations */

/*
**	File types
*/

typedef enum
{
	copy_t, xmit_t, stdin_t
}
	File_t;

FthFD_p	XFiles;			/* Files to be transmitted in place */
FthFD_p*XFend	= &XFiles;	/* Last element in Xfiles */
FthFD_p	CFiles;			/* Files to be copied into message */
FthFD_p*CFend	= &CFiles;	/* Last element in Cfiles */

/*
**	Address types
*/

typedef enum
{
	full_a, user_a, quote_a
}
	Addr_t;

/*
**	Miscellaneous
*/

bool	CFused;			/* True if CommandsFile used */
long	ComLength;		/* Length of message parts described in CommandsFile */
bool	Explain;		/* Explain useage in finish() */
int	FileCount;		/* Number of input files */
bool	FileFirst;		/* True if first arg is name of file */
Passwd	From, To;		/* Details of user */
char *	HomeNode;		/* Name of this node */
int	MesgFd;			/* File descriptor of Message */
Time_t	Time;			/* Used by UniqueName() in Exec...() */
int	Pid;			/* Used by UniqueName() in Exec...() */
char	ProtoType[2];		/* Protocol type for header */

char *	CommandsFile	= WORKDIR(sender.command);
char *	Message		= WORKDIR(sender.message);
FthUlist **LastUser	= &FthUsers;

#define	ADDRESS_LIST	":@"	/* List of legal user/destination separators */
#define	free(A)

void	addfile(), checkXread(), checknodes(), copyfiles(), finish(), getuser(), send();
int	compare(), sigcatch();
char *	getdest();



main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	bool		ftp_handler;
	Handler *	handler;

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	if ( !GetUser(&From, getuid()) )
	{
		Error(From.P_error);
		return;
	}

	FthFrom = From.P_user;

	Pid = getpid();
	Time = time((long *)0);
	HomeNode = NodeName();

	if ( signal(SIGINT, SIG_IGN) != SIG_IGN )
		(void)signal(SIGINT, sigcatch);
	if ( signal(SIGQUIT, SIG_IGN) != SIG_IGN )
		(void)signal(SIGQUIT, sigcatch);
	if ( signal(SIGHUP, SIG_IGN) != SIG_IGN )
		(void)signal(SIGHUP, sigcatch);
	(void)signal(SIGTERM, sigcatch);

	(void)umask(077);

	while
	(
#		if	defined(O_CREAT) && defined(O_FREE)
		(MesgFd = open(
#		else	defined(O_CREAT) && defined(O_FREE)
		(MesgFd = creat(
#		endif	defined(O_CREAT) && defined(O_FREE)
				UniqueName
				(
					Message,
					(long)0,
					Time
				),
#				if	defined(O_CREAT) && defined(O_FREE)
				O_FREE|O_CREAT|O_TRUNC|O_EXCL|O_WRITE,
#				endif	defined(O_CREAT) && defined(O_FREE)
				0600
			  )
		)
		== SYSERROR
	)
		Syserror("Can't creat \"%s\"", Message);

	Explain = true;

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;
			register bool	first = true;

			while ( (c = *++*argv) || first )
			{
				first = false;

				switch ( c )
				{
				case 'A':
					Ack = true;
					continue;

				case 'C':
					CRC = true;
					if ( FileCount )
					{
						Error("'%c' flag must come before any file names", c);
						return;
					}
					continue;

				case 'D':
					NoCall = true;
					continue;

				case 'N':
					NoFtph = true;
					continue;

				case 'R':
					NoRet = true;
					continue;

				case 'S':
					NoOpt = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'a':
					HdrHandler = ++*argv;
					goto break2;

				case 'b':	/* Recognise "user[,user...][@dest[,dest...]]" */
					getuser(++*argv, full_a);
					goto break2;

				case 'd':
					(void)getdest(++*argv);
					goto break2;

				case 'e':
					EnvString = ++*argv;
					goto break2;

				case 'f':
					addfile(++*argv, copy_t);
					goto break2;

				case 'n':
					DataName = ++*argv;
					goto break2;

				case 'o':
					HdrSource = ++*argv;
					if ( !(From.P_flags & P_SU) )
					{
						Error("'%c': no permission", c);
						return;
					}
					goto break2;

				case 'q':	/* Recognise "user[@dest[,dest...]]" */
					getuser(++*argv, quote_a);
					goto break2;

				case 's':
					FthFrom = ++*argv;
					if ( !(From.P_flags & P_SU) )
					{
						Error("'%c': no permission", c);
						return;
					}
					goto break2;

				case 't':
					HdrTtd = ++*argv;
					goto break2;

				case 'u':	/* Recognise "user[,user...]" */
					getuser(++*argv, user_a);
					goto break2;

				case 'x':
					addfile(++*argv, xmit_t);
					goto break2;

				case '\0':
					addfile(NULLSTR, stdin_t);
					goto break2;

				default:
					Mesg("unrecognised flag", "'%c'", c);
					putc('\n', stderr);
					finish(1);
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			register char *	cp;

			if
			(
				!NoFtph
				&&
				(
					(
						NFthUsers == 0
						&&
						Destinations == (Dlist *)0
					)
					||
					(
						FileCount == 0
						&&
						(
							strpbrk(*argv, "@:,") != NULLSTR
							||
							(
								(cp = strchr(*argv, '.')) != NULLSTR
								&&
								strlen(cp+1) > 1
							)
						)
						&&
						access(*argv, 0) == SYSERROR
					)
				)
			)
			{
				if ( NFthUsers == 0 )
				{
					if
					(
						strchr(*argv, '/') != NULLSTR
						&&
						strpbrk(*argv, "@:,") == NULLSTR
					)
					{
						Error("Must specify address first");
						return;
					}
					if ( access(*argv, 0) != SYSERROR )
						FileFirst = true;
				}

				getuser(*argv, full_a);
			}
			else
				addfile(*argv, copy_t);
		}
	}

	checknodes();	/* Set "HdrDest" */

	if
	(
		!(From.P_flags & P_CANSEND)
#		if	LOCALSEND == 1
		&&
		strccmp(HdrDest, HomeNode) != STREQUAL
#		endif	LOCALSEND == 1
	)
	{
		Error("No permission.");
		return;
	}

	if ( FileCount == 0 )
	{
		if ( NFthUsers > 1 && FileFirst )
			Warn("No file specified, reading standard input...");
		addfile(NULLSTR, stdin_t);
	}

	Explain = false;

	if ( NFthUsers == 0 )
		getuser(From.P_name, user_a);

	ftp_handler = true;

	if ( HdrHandler == NULLSTR )
		HdrHandler = FTPHANDLER;

	if ( NoFtph || strcmp(HdrHandler, MAILHANDLER) != STREQUAL )
	{
		if ( !NoFtph && strcmp(HdrHandler, PRINTHANDLER) == STREQUAL )
		{
#			ifdef	PRINTSITES
			register Dlist *	dp;

			for ( dp = Destinations ; dp != (Dlist *)0 ; dp = dp->d_next )
				if ( !InList(dp->d_name, PRINTSITES, NULLSTR) )
#			endif	PRINTSITES
				{
					Error("Illegal print site");
					return;
				}
		}
		else
		if
		(
			(handler = GetHandler(HdrHandler)) == (Handler *)0
			||
			handler->proto_type != FTP
		)
		{
			if
			(
				(
					handler == (Handler *)0
					&&
					!(From.P_flags & P_OTHERHANDLERS)
				)
				||
				(
					handler != (Handler *)0
					&&
					handler->proto_type == STATE_PROTO
				)
			)
			{
				Error("Illegal handler");
				return;
			}

			ftp_handler = false;
		}
		else
		if ( NoFtph )
		{
			Error("Illegal use of flag 'N'");
			return;
		}
	}
	else
	if ( !(From.P_flags & P_SU) )
	{
		Error("Illegal handler");
		return;
	}

	if ( ftp_handler )
		ProtoType[0] = FTP;

	if ( HdrSource == NULLSTR )
		HdrSource = HomeNode;

	HdrEnv = "";

	if ( NoCall )
		HdrEnv = concat(HdrEnv, MakeEnv(ENV_NO_AUTOCALL, NULLSTR, NULLSTR), NULLSTR);

	if ( NoRet )
		HdrEnv = concat(HdrEnv, MakeEnv(ENV_NORET, NULLSTR, NULLSTR), NULLSTR);

	if ( NoOpt )
		HdrEnv = concat(HdrEnv, MakeEnv(ENV_NOOPT, NULLSTR, NULLSTR), NULLSTR);

	if ( EnvString != NULLSTR && EnvString[0] != '\0' )
		HdrEnv = concat
			 (
				HdrEnv,
				MakeEnv(ENV_HANDLER_FLAGS, EnvString, NULLSTR),
				NULLSTR
			 );

	copyfiles();

	send();
}



/*
**	Called from the errors routines to cleanup
*/

void
finish(error)
	int	error;
{
	(void)unlink(Message);

	if ( CFused )
		(void)unlink(CommandsFile);

	if ( Explain )
	{
		Mesg("Usage", Usage, Name);
		putc('\n', stderr);
	}

	(void)exit(error);
}



/*
**	Catch signals and call finish
*/

int
sigcatch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);
	finish(sig);
}



/*
**	A user list must be in the following form:
**
**	user[,user...][<@|:>destination]
**
**	where 'destination' is as defined for getdest().
*/

void
getuser(s, type)
	register char *	s;
	Addr_t		type;
{
	register char *	u;
	register char *	d;
	register FthUlist *up;

	Trace2(1, "getuser \"%s\"", s);

	if ( type != user_a )
	{
		if
		(
			(d = strrpbrk(s, ADDRESS_LIST)) != NULLSTR
			||
			(d = strchr(s, '.')) != NULLSTR
		)
		{
			*d++ = '\0';
			d = getdest(d);
		}
		else
			d = getdest(HomeNode);
	}
	else
		d = NULLSTR;

	do
	{
		if ( type != quote_a )
		{
			if ( (u = strchr(s, ',')) != NULLSTR )
				*u++ = '\0';
		}
		else
			u = NULLSTR;

		if ( *s == '\0' )
			continue;	/* Null user name?? */

		NFthUsers++;
		up = Talloc(FthUlist);
		up->u_next = (FthUlist *)0;
		*LastUser = up;
		LastUser = &up->u_next;
		up->u_dest = d;
		up->u_name = s;

		Trace3(2, "Ulist \"%s:%s\"", s, d);
	}
	while
		( (s = u) != NULLSTR );
}


/*
**	A destination must be in one of the following forms:
**
**	'node[.domain]'					address
**	'*[.domain]'					broadcast
**	'{address|broadcast},{address|broadcast}...'	multicast
**	'multicast!multicast...'			explicit
**
**	An 'explicit' destination is exclusive.
**
**	(See "address.h" for allowable formats.)
*/

char *
getdest(s)
	register char *	s;
{
	extern char *	get_dest();

	Trace2(1, "getdest \"%s\"", s);

	if ( s == NULLSTR || *s == '\0' )
	{
		Error("Null destination?");
		return;
	}

	if
	(
		Destinations != (Dlist *)0
		&&
		(
			Destinations->d_sep == ATYP_EXPLICIT
			||
			strchr(s, ATYP_EXPLICIT) != NULLSTR
		)
	)
	{
		Error("Explicit routing address is exclusive");
		return;
	}

	if ( strpbrk(s, ADDRESS_LIST) != NULLSTR )
	{
		Error("Illegal destination format \"%s\"", s);
		return;
	}

	return get_dest(&Destinations, s, false, '\0');
}



/*
**	Extract simple destination element, or recurse on complex parts.
*/

char *
get_dest(dpp, s, passback, sep)
	Dlist **		dpp;
	register char *		s;
	bool			passback;
	char			sep;
{
	register Dlist *	dp;
	register char *		cp;
	char *			s2;

	Trace3(2, "get_dest \"%s\"%s", s, passback?" (passback)":"");

	if ( (cp = strchr(s, ATYP_EXPLICIT)) != NULLSTR )
	{
		if ( !(From.P_flags & P_EXPLICIT) )
		{
			Error("explicit address disallowed");
			return;
		}
	}
	else
	if ( (cp = strchr(s, ATYP_MULTICAST)) != NULLSTR )
	{
		if ( !(From.P_flags & P_MULTICAST) )
		{
			Error("multicast address disallowed");
			return;
		}
	}
	else
	if ( *s == ATYP_BROADCAST )
	{
		if ( s[1] != '\0' && s[1] != DOMAIN_SEP )
		{
			Error("Illegal broadcast address");
			return;
		}

		if ( !(From.P_flags & P_BROADCAST) )
		{
			Error("broadcast address disallowed");
			return;
		}

node:
		if ( !passback )
		{
			dp = Talloc(Dlist);
			dp->d_part = (Dlist *)0;
			dp->d_prev = dpp;
			if ( (dp->d_next = *dpp) != (Dlist *)0 )
				dp->d_next->d_prev = &dp->d_next;
			*dpp = dp;

			if ( (dp->d_sep = sep) == '\0' )
				dp->d_sep = ATYP_MULTICAST;

			dp->d_name = s;
		}

		return s;
	}
	else
	{
		if ( (cp = FindAlias(s)) != NULLSTR )
			s = cp;
		
		goto node;
	}

	dp = Talloc(Dlist);
	dp->d_part = (Dlist *)0;
	dp->d_prev = dpp;
	if ( (dp->d_next = *dpp) != (Dlist *)0 )
		dp->d_next->d_prev = &dp->d_next;
	*dpp = dp;

	if ( sep == '\0' )
		sep = *cp;

	dp->d_sep = sep;

	if ( *cp == sep )
	{
		*cp++ = '\0';

		s2 = get_dest(&dp->d_next, cp, false, sep);
	}
	else
		s2 = NULLSTR;

	dp->d_name = get_dest(&dp->d_part, s, true, '\0');

	if ( s2 != NULLSTR )
	{
		s = cp = Malloc(strlen(dp->d_name) + strlen(s2) + 2);
		cp = strcpyend(cp, dp->d_name);
		*cp++ = dp->d_sep;
		(void)strcpy(cp, s2);
	}
	else
		s = dp->d_name;

	Trace3(2, "get_dest returns \"%s\" at \"%s\"", s, dp->d_name);

	return s;
}



/*
**	Re-create complex address string from destination list.
*/

char *
makedest(dp, cp)
	register Dlist *dp;
	register char *	cp;
{
	Trace3(2, "makedest at \"%c%s\"", dp->d_sep, dp->d_name);

	do
	{
		if ( (*cp++ = dp->d_sep) == '\0' )
			cp--;

		if ( dp->d_part )
			cp = makedest(dp->d_part, cp);
		else
			cp = strcpyend(cp, dp->d_name);
	}
		while ( (dp = dp->d_next) != (Dlist *)0 );
	
	return cp;
}



/*
**	Make HdrDest list.
*/

void
checknodes()
{
	Trace1(1, "checknodes");

	if ( Destinations == (Dlist *)0 )
		HdrDest = HomeNode;
	else
		makedest(Destinations, HdrDest = Malloc(checklist(Destinations)));

	Trace2(1, "HdrDest \"%s\"", HdrDest);
}



/*
**	Check nodes are unique.
*/

int
checklist(dp)
	Dlist *		dp;
{
	register Dlist *np;
	register int	i;
	int		count;
	int		length;

	count = 0;
	length = 1;

	np = dp; 
	do
	{
		count++;
		length++;

		if ( np->d_part != (Dlist *)0 )
			length += checklist(np->d_part);
		else
			length += strlen(np->d_name);
	}
		while ( (np = np->d_next) != (Dlist *)0 );

	if ( (i = count) > 1 )
	{
		register Dlist**dpp;

		dpp = (Dlist **)Malloc(i * sizeof(Dlist *));

		np = dp;
		do
			dpp[--i] = np;
		while
			( (np = np->d_next) != (Dlist *)0 );

		if ( dp->d_sep != ATYP_EXPLICIT )
			qsort((char *)dpp, count, sizeof(Dlist *), compare);

		i = count;

		while ( --i > 0 )
			if ( strccmp(dpp[i-1]->d_name, dpp[i]->d_name) == STREQUAL )
			{
				/*
				**	Remove unnecessary node
				*/

				np = dpp[i];

				Trace2(2, "remove dup node \"%s\"", np->d_name);

				if ( (*np->d_prev = np->d_next) != (Dlist *)0 )
					np->d_next->d_prev = np->d_prev;

				length -= strlen(np->d_name);
				free((char *)np);

				count--;
			}

		if ( count == 1 )
			dpp[0]->d_sep = '\0';

		free((char *)dpp);
	}
	else
		dp->d_sep = '\0';

	Trace3(2, "checklist returns %d, at \"%s\"", length, dp->d_name);

	return length;
}



/*
**	Compare two destination names.
*/

int
compare(dp1, dp2)
	char *	dp1;
	char *	dp2;
{
	return strccmp((*(Dlist **)dp2)->d_name, (*(Dlist **)dp1)->d_name);
}



/*
**	Add a file name into appropriate list, and extract details.
*/

void
addfile(file, type)
	char *		file;
	File_t		type;
{
	register FthFD_p fp;
	struct stat	statb;

	if ( type != stdin_t && (file == NULLSTR || file[0] == '\0') )
	{
		Error("Null file name in argument list");
		return;
	}

	if ( type == xmit_t && file[0] != '/' )
	{
		Error("Need full path name for directly transmitted file");
		return;
	}

	Explain = false;

	fp = Talloc(FthFDesc);
	fp->f_next = (FthFDesc *)0;

	if ( (fp->f_name = file) == NULLSTR )
	{
		if ( fstat(0, &statb) == SYSERROR )
			Syserror("Can't fstat stdin");

		if ( (statb.st_mode & FTH_MODES) == 0 )
			statb.st_mode |= 0600;	/* Silly pipe? */

		if ( statb.st_mtime == 0 )
			statb.st_mtime = Time;	/* ditto */
	}
	else
	if
	(
		access(file, 04) == SYSERROR
		||
		stat(file, &statb) == SYSERROR
	)
		Syserror("Can't read \"%s\"", file);

	if ( (statb.st_mode & S_IFMT) == S_IFDIR )
	{
		free((char *)fp);
		Warn("Directory \"%s\" ignored", file);
	}
	else
	{
		fp->f_time = statb.st_mtime;
		fp->f_mode = statb.st_mode & FTH_MODES;
		fp->f_length = statb.st_size;

		if ( type != xmit_t )
		{
			*CFend = fp;
			CFend = &fp->f_next;
		}
		else
		{
			fp->f_mode |= FTH_NOT_IN_MESG;
			*XFend = fp;
			XFend = &fp->f_next;
		}
	}

	FileCount++;

	Explain = true;
}




/*
**	Read files, create CRC, add to command file, or copy into message.
*/

void
copyfiles()
{
	register FthFD_p	fp;
	register int		fd;
	register int		n;
	FthFD_p *		fpp;
	ComHead			commands;
	char			buf[BUFSIZ];

	fpp = &FthFiles;

	/*
	**	First, the in-place files.
	*/

	FreeCom(&commands, init_ch);

	for ( fp = XFiles ; fp != (FthFD_p)0 ; fp = fp->f_next )
	{
		AddCom(&commands, fp->f_name, (long)0, fp->f_length);

		if ( CRC )
		{
			while ( (fd = open(fp->f_name, O_READ)) == SYSERROR )
				Syserror("can't read \"%s\"", fp->f_name);

			while ( (n = read(fd, buf, sizeof buf)) > 0 )
				DataCrc = acrc(DataCrc, buf, n);

			if ( n == SYSERROR )
				Syserror("Can't read \"%s\"", fp->f_name);

			(void)close(fd);
		}
	}

	if ( ComLength = commands.ch_size )
	{
		CFused = true;

		while
		(
			(fd = creat
			      (
				UniqueName
				(
					CommandsFile,
					ComLength,
					Time
				),
				0600
			      )
			)
			== SYSERROR
		)
			Syserror("Can't creat \%s\"", CommandsFile);
		
		(void)WriteCom(&commands, fd);
		(void)close(fd);

		FreeCom(&commands, free_ch);

		*fpp = XFiles;
		fpp = XFend;
	}

	/*
	**	Next, the in-message files.
	*/

	*fpp = CFiles;

	for ( fp = CFiles ; fp != (FthFD_p)0 ; fp = fp->f_next )
	{
		if ( fp->f_name == NULLSTR )
		{
			if ( (fp->f_name = DataName) == NULLSTR )
				fp->f_name = "stdin";

			fd = 0;
		}
		else
		{
			while ( (fd = open(fp->f_name, O_READ)) == SYSERROR )
				Syserror("can't read \"%s\"", fp->f_name);

			if ( DataName != NULLSTR && FileCount == 1 )
				fp->f_name = DataName;
		}

		fp->f_length = 0;

		while ( (n = read(fd, buf, sizeof buf)) > 0 )
		{
			register char *	cp;
			register int	r;

			fp->f_length += n;

			if ( CRC )
				DataCrc = acrc(DataCrc, buf, n);

			cp = buf;

			while ( (r = write(MesgFd, cp, n)) != n )
			{
				if ( r == SYSERROR )
					Syserror("Can't write \"%s\"", Message);
				else
				{
					cp += r;
					n -= r;
				}
			}
		}

		if ( n == SYSERROR )
			Syserror("Can't read \"%s\"", fp->f_name);

		if ( fd != 0 )
			(void)close(fd);

		DataLength += fp->f_length;
	}
}



/*
**	Write headers and call receiver
*/

void
send()
{
	register int	size;
	register char**	cpp;
	char *		args[8];

	if ( !NoFtph )
	{
		SetFthTo();

		Trace2(1, "FthTo \"%s\"", FthTo);

		SetFthFiles();

		Trace2(1, "FthFdescs \"%s\"", FthFdescs);

		while ( (size = WriteFtHeader(MesgFd, DataLength, Ack, CRC)) == SYSERROR )
			Syserror("Can't write \"%s\"", Message);

		DataLength += size;
	}

	HdrSubpt = ProtoType;

	while ( WriteHeader(MesgFd, DataLength, 0) == SYSERROR )
		Syserror("Can't write \"%s\"", Message);

	(void)close(MesgFd);

	(void)chown(Message, ACSNETUID, ACSNETGID);

	if ( CFused )
		(void)chown(CommandsFile, ACSNETUID, ACSNETGID);

#ifndef	NICEDAEMON
#define	NICEDAEMON	0
#endif	NICEDAEMON

#	if	SYSTEM > 0
	(void)nice(NICEDAEMON-nice(0));
#	else	SYSTEM > 0
	if ( getuid() == 0 )
	{
		(void)nice(-40);
		(void)nice(20+NICEDAEMON);
	}
#	endif	SYSTEM > 0

	SetUser(ACSNETUID, ACSNETGID);

	(void)signal(SIGINT, SIG_IGN);
	(void)signal(SIGQUIT, SIG_IGN);
	(void)signal(SIGHUP, SIG_IGN);

	cpp = args;
	*cpp++ = RECEIVER;
#	ifdef	DEBUG
	if ( Traceflag )
	{
		char	traceflag[4];

		(void)sprintf(traceflag, "-T%d", Traceflag);
		*cpp++ = traceflag;
	}
#	endif	DEBUG
	*cpp++ = concat("-Lh", HomeNode, NULLSTR);
	if ( CFused )
	{
		char	comlength[LENGTH_SIZE+1];

		checkXread();
		(void)sprintf(comlength, "-d%ld", ComLength);
		*cpp++ = concat("-c", CommandsFile, NULLSTR);
		*cpp++ = comlength;
	}
	*cpp++ = Message;
	*cpp = NULLSTR;

	(void)close(0);
	(void)open("/dev/null", O_READ);

	for ( size = 3 ; close(size) != SYSERROR || size < 5 ; size++ );

	for ( ;; )
	{
		(void)execve(args[0], args, StripEnv());
		Syserror("Can't execve \"%s\"", args[0]);
	}
}



/*
**	Check ACSNETUID has read permission on all the XFiles.
*/

void
checkXread()
{
	register FthFD_p	fp;

	for ( fp = XFiles ; fp != (FthFD_p)0 ; fp = fp->f_next )
	{
		if ( access(fp->f_name, 04) == SYSERROR )
			Syserror("daemon can't read \"%s\"", fp->f_name);
		
		if ( XFend == &fp->f_next )
			break;
	}
}
