/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID %W% %E%
*/

/*
**	Protocol handler driver
*/

#include	"global.h"
#include	"command.h"
#include	"debug.h"

#if	OLD_CC != 1 && CEMANTICS != 1
/*
**	This appears to be necessary as some "cc"s don't allow
**	addresses of functions returning void to be assigned!
*/
#define	void	int
#endif	OLD_CC

#include	<setjmp.h>
#include	<signal.h>
#include	<errno.h>

#include	"Channel.h"

#include	"daemon.h"
#undef	Extern
#define	Extern
#include	"AQ.h"
#include	"CQ.h"
#include	"PQ.h"
#include	"Stream.h"
#include	"driver.h"

Channel		Channels[NSTREAMS];
CQhead		ControlQ[NSTREAMS];
AQhead		ActionQ			= { (AQl_p)0, &ActionQ.aq_first };

jmp_buf		IOerrorbuf;		/* Global I/O error recovery */
jmp_buf		QTimeouts;		/* Global action queue timeout recovery */

static jmp_buf	RTimeouts;		/* Read timeout receovery */
static unsigned	ElapsedTime;
static bool	inQ;			/* True if in queue handlers */

#if	DEBUG >= 1
#include	<stdio.h>
#include	"Debug.h"
#endif



void
driver()
{
	register int	i;
	Time_t		block_time = 0;
	int		block_count = 0;

	NNstate.linkstate = LINK_DOWN;
	NNstate.maxspeed = 0;
	NNstate.allbytes = 0;
	NNstate.allmessages = 0;
	inByteCount = 0;
	outByteCount = 0;

	/*
	**	Initialise queues
	*/

	for ( i = 0 ; i < NSTREAMS ; i++ )
		ControlQ[i].cq_last = &ControlQ[i].cq_first;

	for ( i = 0 ; i < NPKTQ ; i++ )
		PktQ[i].pq_last = &PktQ[i].pq_first;

	/*
	**	Initialise function pointers for handler
	*/

	PqPkt = qPkt;
	PqCpkt = qCpkt;
	PfillPkt = fillPkt;
	PrecvData = recvData;
	PrecvControl = recvControl;
	PrTimeout = rTimeout;
	PrReset = rReset;
	PxReset = xReset;
	PRread = Cook ? RCread : Rread;

	/*
	**	Catch I/O errors (1=read, 2=write).
	**	Close link.
	*/

	if ( i = setjmp(IOerrorbuf) )
	{
		ALARM_OFF();

		Report3("%s error (%d) on remote", i==1?"read":"write", errno);

		if ( BatchMode )
		{
			FinishReason = "Remote I/O ERROR";
			return;
		}

		/*
		**	Flush the actions
		*/

		{
			register AQl_p	aqlp;

			inQ = true;
			NoAction = true;

			while ( (aqlp = ActionQ.aq_first) != (AQl_p)0 )
			{
				ActionQ.aq_first = aqlp->aql_next;
				aqlp->aql_next = AQfreelist;
				AQfreelist = aqlp;
				(*aqlp->aql_funcp)(aqlp->aql_arg.qu_arg);
			}

			ActionQ.aq_last = &ActionQ.aq_first;

			NoAction = false;
			inQ = false;

			Waiting = true;
			qAction(WaitHandler, (AQarg)LastTime+3);
		}

		/*
		**	Flush the output.
		*/

		for ( i = 0 ; i < NPKTQ ; i++ )
		{
			register PQl_p	pqlp;

			while ( (pqlp = PktQ[i].pq_first) != (PQl_p)0 )
			{
				PktQ[i].pq_first = pqlp->pql_next;
				pqlp->pql_next = PQfreelist;
				PQfreelist = pqlp;
			}

			PktQ[i].pq_last = &PktQ[i].pq_first;
		}

		/*
		**	Flush the control queues.
		*/

		for ( i = 0 ; i < NSTREAMS ; i++ )
		{
			register CQl_p	cqlp;

			while ( (cqlp = ControlQ[i].cq_first) != (CQl_p)0 )
			{
				ControlQ[i].cq_first = cqlp->cql_next;
				cqlp->cql_next = CQfreelist;
				CQfreelist = cqlp;
			}

			ControlQ[i].cq_last = &ControlQ[i].cq_first;
		}

		/*
		**	Shut down the link
		*/

		NewState(LINK_DOWN, false);

		close_remote();

		(void)sleep(4 * IDLE_TIMEOUT);	/* Make sure other end gets the message */
	}

	/*
	**	Initialise protocol parameters
	*/

	PItimo = IDLE_TIMEOUT;
	PprotoT = (int)UseCrc;
	Pnchans = Nstreams;
	Pfchan = Fstream;
	PXsize = PktZ;
	PRtimo = RECEIVE_TIMEOUT;
	Pnbufs = Nbufs;

	/*
	**	Initialise protocol
	*/

	if ( !Pinit() )
	{
		Fatal1("bad protocol initialisation");
		return;
	}

	PktZ = PXsize;	/* Remember maximum allowed */

	SetSpeed();

	NNstate.inpkts = 0;
	NNstate.outpkts = 0;

	/*
	**	On "Rread" timeout call Ptimo
	*/

	if ( setjmp(RTimeouts) )
	{
		Trace2(1, "READ TIMEOUT %u", ElapsedTime);

		Pflush();

Timeout:
		DODEBUG(if(Traceflag){Mesg("TIME", "\n");(void)fflush(stderr);});

		if ( block_time > 0 && (block_time + PRtimo) <= LastTime )
		{
			block_time = 0;

			if ( ++block_count > 10 )
			{
				block_count = 0;
				errno = 0;
				longjmp(IOerrorbuf, 2);
			}
		}

		if
		(
			BatchMode
			&&
			NNstate.activetime > 120
			&&
			(NNstate.allbytes/NNstate.activetime) < MinSpeed
		)
		{
			FinishReason = "Remote SLOW";
			return;
		}

		if ( block_time == 0 && Ptimo((Timo_t)ElapsedTime) )
		{
			NewState(LINK_DOWN, false);

			if ( BatchMode )
			{
				FinishReason = "Remote TIMEOUT";
				return;
			}
		}
		else
		if ( NNstate.inpkts )
			NewState(LINK_UP, false);

		if ( (UpdateTime += ElapsedTime) >= UPDATE_TIME || ContinuousUpdate )
			Update(up_date);
	}
	else
	{
		/*
		**	Open and initialise the link.
		*/

		char	c[1];

		(void)alarmcatch(0);

		inQ = true;

		if ( setjmp(QTimeouts) )
			NewState(LINK_DOWN, false);

		ALARM_ON(5 * IDLE_TIMEOUT);

		if ( !open_remote() )
		{
			ALARM_OFF();
			FinishReason = "Remote open ERROR";
			return;
		}

		/*
		**	Flush/Sync...
		*/

		if ( !setjmp(QTimeouts) )
		{
			ALARM_ON(2);
			if ( BatchMode )
				pause();
			else
				while ( read(RemoteFd, c, sizeof c) == sizeof c )
					;
		}
	}

	ElapsedTime = 0;

	/*
	**	Driver loop
	*/

	for(;;)
	{
		inQ = true;

		if ( block_time == 0 )
		for ( ;; )
		{
			/*
			**	Transmit all queued packets,
			**	 unless both transmitting and receiving,
			**	 in which case transmit up to first data packet only.
			*/

			register PQl_p	pqlp;
			register int	n;
			register int	outcount = 0;

			if ( setjmp(QTimeouts) )
			{
				Report1("write BLOCKED");
				block_time = time((long *)0);
				break;
			}

			for ( i = 0 ; i < NPKTQ ; i++ )
			while ( (pqlp = PktQ[i].pq_first) != (PQl_p)0 )
			{
				register char *	pktp = pqlp->pql_pktp;
				register int	size = pqlp->pql_size;

				DODEBUG(Logpkt((Pkt_p)pktp, PLOGSEND));

				if ( !Alarm )
					ALARM_ON(15);

				while ( (n = (*Write)(RemoteFd, pktp, size)) != size )
				{
					if ( n <= 0 )
					{
						if ( n == SYSERROR )
						{
#							if	DEBUG >= 1
							if ( errno == EINTR )
								continue;
#							endif	DEBUG >= 1
							longjmp(IOerrorbuf, 2);
						}
						longjmp(QTimeouts, 1);
					}

					pktp += n;
					size -= n;
				}

				Trace2(2, "Write %d", pqlp->pql_size);

				NNstate.outpkts++;
				outcount++;
				block_count = 0;

				if ( (PktQ[i].pq_first = pqlp->pql_next) == (PQl_p)0 )
					PktQ[i].pq_last = &PktQ[i].pq_first;

				pqlp->pql_next = PQfreelist;
				PQfreelist = pqlp;

				if
				(
					Transmitting
					&&
					Receiving
					&&
					i != PCNTRLQ
				)
					goto break2;
			}

			if ( Receiving && outcount == 0 && Plastidle == 0 )
			{
				Pidle();
				continue;
			}
break2:
			ALARM_OFF();
			break;
		}

		/*
		**	Perform any queued actions
		*/

		{
			register AQl_p	aqlp;
			register AQl_p	next;

			if ( (aqlp = ActionQ.aq_first) != (AQl_p)0 )
			{
				ActionQ.aq_first = (AQl_p)0;
				ActionQ.aq_last = &ActionQ.aq_first;

				do
				{
					next = aqlp->aql_next;
					aqlp->aql_next = AQfreelist;
					AQfreelist = aqlp;

					(*aqlp->aql_funcp)(aqlp->aql_arg.qu_arg);
				}
					while ( (aqlp = next) != (AQl_p)0 );
			}
		}

		inQ = false;

		/*
		**	Call protocol receiver to read remote
		*/

		Precv();

		/*
		**	Perform timeout scan if necessary
		*/

		{
			register unsigned interval;
			register Time_t	thistime;

			thistime = time((long *)0);

			if ( interval = thistime - LastTime )
			{
				LastTime = thistime;

				if ( (ElapsedTime += interval) >= ScanRate )
					goto Timeout;
			}
		}
	}
}



/*
**	Catch alarm signals from system
*/

int
alarmcatch(sig)
	int		sig;
{
	register Time_t	thistime;

	(void)signal(SIGALRM, alarmcatch);

	thistime = time((long *)0);

	Trace5(1, "alarmcatch(%d), time=%lu, Elapsed=%u, Last=%lu", sig, thistime, ElapsedTime, LastTime);

	ElapsedTime += thistime - LastTime;
	LastTime = thistime;
	Alarm = false;

	if ( sig )
		if ( inQ )
			longjmp(QTimeouts, 1);
		else
			longjmp(RTimeouts, 1);
}



/*
**	Recalculate PRtimo if set speed of link differs from the
**	effective speed by more than 20%.
*/

void
SetSpeed()
{
	register int	xmax;
	register int	npkts;

	if ( !NoAdjust && NNstate.activetime > 60 )
	{
		register int	speed;
		register int	diff;

		speed = NNstate.allbytes/NNstate.activetime;

		if ( (diff = speed-Speed) < 0 )
			diff = -diff;

		if ( diff > Speed/5 )
			Speed += (speed-Speed) / 5;
	}

	xmax = min(PXmax, PktZ);

	if ( (npkts = (Transmitting + Receiving) * Pnbufs) == 0 )
		npkts = Nstreams - Fstream;

	for ( ;; )
	{
		PRtimo = (
				npkts			/* Number of packets in flight ahead */
				* (PXsize+2*Poverhead)	/* Size of packet + ack */
				+ Speed-1		/* Round up */
			 )
			 / Speed			/* Divide by speed */
			 + 1;				/* Add safety factor */

		if ( NoAdjust )
			break;

		if ( PRtimo > IDLE_TIMEOUT )
		{
			if ( PXsize <= MINPKTSIZE )
			{
				PXsize = MINPKTSIZE;
				if ( PRtimo > IDLE_TIMEOUT )
					PRtimo = IDLE_TIMEOUT;
				break;
			}

			PXsize /= 2;
		}
		else
		if ( PRtimo < RECEIVE_TIMEOUT )
		{
			if ( PXsize >= xmax )
			{
				PRtimo = RECEIVE_TIMEOUT;
				PXsize = xmax;
				break;
			}

			PXsize *= 2;
		}
		else
			break;
	}

	PItimo = (IDLE_SCANRATE-4) + PRtimo * 2;

	if ( (Transmitting + Receiving) == 0 )
		ScanRate = IDLE_SCANRATE;
	else
	{
		/*
		**	Set ScanRate large enough to receive whole packet.
		*/

		ScanRate = PXsize + Poverhead;
		if ( Cook )
			ScanRate *= 2;
		if ( (ScanRate = (ScanRate+Speed-1)/Speed + 1) < ACTIVE_SCANRATE )
			ScanRate = ACTIVE_SCANRATE;

		ScanRate += IntraPktDelay;
	}

	NNstate.recvtimo = PRtimo;
	NNstate.packetsize = PXsize;
	NNstate.speed = Speed;
	NNstate.nbufs = Pnbufs;
}
