/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "88/11/04 @(#)stop.c	1.23";

/*
**	Stop messages awaiting transmission.
**
**	SETUID ==> ACSNETUID.
*/

char *	Usage	= "[-[A][M][R][S][T<level>][V][W][Y]] \\\n\
	[-f<source>] [-u<user>] [link|alias|node ...]";

#define	FILE_CONTROL
#define	STAT_CALL
#define	STDIO

#include	<sys/param.h>	/* For BSHIFT iff SYSTEM > 0 */

#include	"global.h"

#include	"Passwd.h"
#include	"command.h"
#include	"debug.h"
#include	"ftheader.h"
#include	"handlers.h"
#include	"header.h"
#include	"spool.h"
#include	"state.h"
#include	"sub_proto.h"
#include	"Stream.h"

#include	<ndir.h>
#include	<errno.h>
#ifndef	SIG_IGN
#include	<signal.h>
#endif


/*
**	Parameters set from arguments
*/

bool	All;				/* Look at all messages */
bool	MessageDetails;			/* Print header details from messages */
char *	Name;				/* Program invoked name */
char *	RHandler;			/* Restrict messages to those for this handler */
char *	Reason;				/* Reason for being stopped */
bool	Reroute;			/* Look for re-routed messages */
char *	SourceNode;			/* Look for messages from a particular source */
int	Traceflag;			/* Trace level */
char *	User;				/* Different user to look for */
bool	Verbose;			/* Print all details from ftp header */
bool	Warnings;			/* Whinge */
bool	Yes;				/* Be obsequious */

/*
**	List for found links.
*/

typedef struct LLel *	LLl_p;

typedef struct LLel
{
	LLl_p	lql_next;
	char *	lql_link;
	ino_t	lql_ino;
}
	LLel;

LLl_p	LinkList;			/* Head of linked list */
LLl_p *	LinkVec;			/* Vector of sorted links */
int	NLinks;				/* No. of links found */

/*
**	Structure to remember "unlink" file names.
*/

typedef struct UnlinkN *UN_p;
typedef struct UnlinkN
{
	UN_p	un_next;
	char *	un_name;
}
	UnlinkN;

UN_p	UnFreelist;
UN_p	UnList;

/*
**	Miscellaneous info.
*/

char *	Format		= "%-7s from %-8.8s to %-13.13s at %-13.13s %10ld bytes\n";
char *	FormatV		= "%s from %s to %s at %s  %ld bytes\n";
char *	HeaderFile;			/* Current message header */
char *	HomeNode;			/* Here */
char *	Home_Address;			/* Our address */
char	LinkName[132];			/* Link name for printing */
bool	LinkPrinted;			/* True after LinkName printed */
bool	Local;				/* True if current message originated here */
Passwd	Me;				/* My name */
long	MesgDataSize;			/* Size of data part */
char *	MesgFile;			/* Data part of message (if any) */
long	MesgHdrSize;			/* Size of header */
long	MesgLength;			/* Total size of message */
int	Pid;				/* Used by UniqueName() in Exec...() */
char *	Spooldir	= SPOOLDIR();
int	Stopped;			/* Messages stopped */
Time_t	Time;				/* Time of invocation */
long	Time_to_die;			/* From timeout command */
Time_t	WaitTime;			/* Time message has been in queue */

#define	Fprintf		(void)fprintf
#define	Fflush		(void)fflush
#define	dots(A)		(A[0]=='.'&&(A[1]==0||(A[1]=='.'&&A[2]==0)))
#define	MSGINDENT	(6+7+1)

/*
**	Routines
*/

void	ListLink(), Return(), SortLinks(), Stopem(), print_header(), usage();
bool	FindLinks(), query(), searchstatus();
int	compare(), msgcmp();
bool	read_com(), read_header(), read_ftp();



int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	register int	i;
	char		buf[BUFSIZ];

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Pid = getpid();
	Time = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'A':
					All = true;
					continue;

				case 'M':
					MessageDetails = true;
					continue;

				case 'R':
					Reroute = true;
					continue;

				case 'S':
					RHandler = STATEHANDLER;
					All = true;
					continue;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'V':
					Verbose = true;
					continue;

				case 'W':
					Warnings = true;
					continue;

				case 'Y':
					Yes = true;
					continue;

				case 'a':
					RHandler = ++*argv;
					goto break2;

				case 'f':
					SourceNode = ++*argv;
					goto break2;

				case 'r':
					Reason = ++*argv;
					goto break2;

				case 'u':
					User = ++*argv;
					goto break2;

				default:
					usage("unrecognised flag '%c'", (char *)c);
					exit(1);
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			char *	link = *argv;

			for ( i = 0 ; ; i++ )
			{
				char *		linkp = concat(Spooldir, link, NULLSTR);
				struct stat	statb;

				if
				(
					link[0] != '\0'
					&&
					!dots(link)
					&&
					stat(linkp, &statb) != SYSERROR
					&&
					(statb.st_mode&S_IFMT) == S_IFDIR
				)
				{
					ListLink(linkp, statb.st_ino);
					break;
				}

				free(linkp);

				if ( i == 0 )
				{
					NodeLink nl;

					if ( (link = FindAlias(*argv)) == NULLSTR )
						link = *argv;

					if ( FindAddress(link, &nl) )
					{
						link = nl.nl_name;
						continue;
					}
				}

				Error("unknown link \"%s\"", *argv);
				exit(1);
			}
		}
	}

	if ( NLinks == 0 && !FindLinks(Spooldir) )
	{
		Error("no network links found!");
		exit(1);
	}

	if ( !GetUser(&Me, getuid()) )
	{
		Error("passwd error for uid %d: \"%s\"", Me.P_uid, Me.P_error);
		exit(1);
	}

	if ( !(Me.P_flags & P_SU) && (All || User != NULLSTR) )
	{
		Error("No permission.");
		exit(1);
	}

	if ( User != NULLSTR && !GetUid(&Me, User) )
	{
		Error("passwd error for user \"%s\": \"%s\"", User, Me.P_error);
		exit(1);
	}

	if ( Reroute )
		ListLink(concat(REROUTEDIR(), ".", NULLSTR), (ino_t)0);

	SortLinks();

	HomeNode = NodeName();
	Home_Address = concat(HomeNode, Hierarchy(), NULLSTR);

	setbuf(stdout, buf);

	for ( i = 0 ; i < NLinks ; i++ )
		Stopem(LinkVec[i]->lql_link);

	Fprintf(stdout, "%d message%s stopped.\n", Stopped, Stopped==1?"":"s");

	exit(0);
}



/*
**	Cleanup for error routines
*/

void
finish(error)
	int	error;
{
	(void)exit(error);
}



/*
**	Explain usage
*/

#if	VARARGS
void
usage(va_alist)
	va_dcl
{
	va_list		ap;
	register char 	*s;

	va_start(ap);
	s = va_arg(ap, char *);
	VMesg("bad arguments", s, ap);
	va_end(ap);
	(void)fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}
#else
/*VARARGS1*/
void
usage(s, a1)
	char *	s;
	char *	a1;
{
	Mesg("bad arguments", s, a1);
	Fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}
#endif



/*
**	Queue a network link directory.
*/

void
ListLink(path, ino)
	char *		path;
	ino_t		ino;
{
	register LLl_p	hllp;

	hllp = Talloc(LLel);

	hllp->lql_link = path;
	hllp->lql_ino = ino;

	hllp->lql_next = LinkList;
	LinkList = hllp;

	NLinks++;

	Trace2(1, "list link \"%s\"", hllp->lql_link);
}




/*
**	Search spooldir for link directories.
*/

bool
FindLinks(path)
	char *				path;
{
	register DIR *			dirp;
	register struct direct *	direp;
	bool				found = false;

	Trace2(1, "find links in \"%s\"", path);

	if ( (dirp = opendir(path)) == NULL )
	{
		Syserror("cannot read \"%s\"", path);
		return found;
	}

	while ( (direp = readdir(dirp)) != NULL )
	if ( direp->d_name[0] != WORKFLAG && !dots(direp->d_name) )
	{
		register LLl_p	hllp;
		struct stat	dstat;
		char *		newpath;

		Trace2(2, "entry \"%s\"", direp->d_name);

		for ( hllp = LinkList ; hllp != (LLl_p)0 ; hllp = hllp->lql_next )
			if ( hllp->lql_ino == direp->d_ino )
				break;

		if ( hllp == (LLl_p)0 )
		{
			newpath = concat(path, direp->d_name, NULLSTR);

			if
			(
				stat(newpath, &dstat) != SYSERROR
				&&
				(dstat.st_mode & S_IFMT) == S_IFDIR
			)
			{
				ListLink(newpath, direp->d_ino);
				found = true;
			}
			else
				free(newpath);
		}
	}

	closedir(dirp);

	return found;
}


/*
**	Sort the links into alphabetic order
*/

void
SortLinks()
{
	register LLl_p	lp;
	register LLl_p *lpp;

	LinkVec = lpp = (LLl_p *)Malloc(sizeof(LLl_p) * NLinks);

	for ( lp = LinkList ; lp != (LLl_p)0 ; lp = lp->lql_next )
		*lpp++ = lp;

	DODEBUG(if((lpp-LinkVec)!=NLinks)Fatal1("bad NLinks"));

	Trace2(1, "found %d links", NLinks);

	if ( NLinks > 1 )
		qsort((char *)LinkVec, NLinks, sizeof(LLl_p), compare);
}



/*
**	Alpha compare
*/

int
compare(lpp1, lpp2)
	LLl_p *lpp1;
	LLl_p *lpp2;
{
	return strcmp((*lpp1)->lql_link, (*lpp2)->lql_link);
}



/*
**	Search a directory for appropriate N-N command files,
**	print details, and prompt for action.
*/

void
Stopem(dir)
	char *				dir;
{
	register DIR *			dirp;
	register struct direct *	direp;
	register int			fd;
	char *				link;

	Trace2(1, "Stopem \"%s\"", dir);

	if ( (link = strrchr(dir, '/')) != NULLSTR && link[1] != '\0' )
		link++;
	else
		link = dir;

	(void)sprintf(LinkName, "Link to %s:-\n", link);
	LinkPrinted = false;

	if ( (dirp = opendir(dir)) == NULL )
	{
		Syserror("Can't read \"%s\"", dir);
		return;
	}

	while ( (direp = readdir(dirp)) != NULL )
	{
		register char *	fp;
		register UN_p	unp;
		Time_t		mtime;
		bool		val;
		bool		timed_out;

		/*
		**	Find valid command file.
		*/

		switch ( direp->d_name[0] )
		{
		default:
			continue;

		case SMALL_ID:
		case MEDIUM_ID:
		case LARGE_ID:
			break;
		}

		Trace2(2, "found \"%s\"", direp->d_name);

		fp = concat(dir, "/", direp->d_name, NULLSTR);

		if ( (fd = open(fp, O_RDWR)) == SYSERROR )
		{
			if ( Warnings )
				SysWarn("Can't read commands file \"%s\"", fp);

			free(fp);
			continue;
		}

		MesgLength = 0;
		MesgDataSize = 0;
		MesgHdrSize = 0;

		if ( MesgFile != NULLSTR )
			free(MesgFile);
		MesgFile = NULLSTR;

		if ( HeaderFile != NULLSTR )
			free(HeaderFile);
		HeaderFile = NULLSTR;

		Time_to_die = 0;
		Time = time((long *)0);

		if
		(
			!ReadCom(fd, &mtime, read_com)
			||
			HeaderFile == NULLSTR
		)
		{
			if ( Warnings )
				Warn("bad commands file \"%s\"", fp);

			val = false;
		}
		else
		if ( val = read_header() )
		{
			Time_t	ttd;

			ttd = mtime + Time_to_die;

			if
			(
				(timed_out = (bool)(Time_to_die>0 && Time>ttd))
				&&
				MessageDetails
			)
			{
				Fprintf
				(
					stdout,
					"%*sTime-to-die: %.15s\n",
					MSGINDENT, "",
					ctime(&ttd)+4
				);
			}
		}

		if ( val && query(timed_out) )
		{
			char *	statusfile = concat(dir, "/", STATUSFILE, NULLSTR);

			if ( !timed_out )
			{
				ComHead	comhead;

				if ( !Local )
					Return(link);

				FreeCom(&comhead, init_ch);
				AddCom(&comhead, NULLSTR, (long)1, (long)0);	/* Timeout in 1 second */
				(void)WriteCom(&comhead, fd, fp);		/* 'fd' is at EOF after "ReadCom()" above */
				FreeCom(&comhead, free_ch);

				if ( searchstatus(statusfile, direp->d_name) )
					Warn("Might not be stopped: message is being transmitted.");

				Stopped++;
			}

			(void)unlink(fp);	/* Unlink command file */

			for ( unp = UnList ; unp != (UN_p)0 ; unp = unp->un_next )
			{
				struct stat	mstat;

				if ( stat(unp->un_name, &mstat) != SYSERROR )
				{
					register int	f;

					/*
					**	Truncate message
					*/

					if
					(
						Local && mstat.st_nlink == 1
						&&
						(f = creat(unp->un_name, 0600)) != SYSERROR
					)
						(void)close(f);

					(void)unlink(unp->un_name);
				}
			}

			free(statusfile);
		}

		while ( (unp = UnList) != (UN_p)0 )
		{
			UnList = unp->un_next;
			unp->un_next = UnFreelist;
			UnFreelist = unp;
			free(unp->un_name);
		}

		(void)close(fd);
		free(fp);
	}

	closedir(dirp);
}



/*
**	Search NNdaemon status file for transmitting command file name.
*/

bool
searchstatus(statusfile, commandfile)
	char *		statusfile;
	char *		commandfile;
{
	register int	i;
	register int	fd;
	static NN_state	NNstate;
	bool		exists;
	static char *	laststfile;

	Trace2(1, "searchstatus \"%s\"", statusfile);

	for ( i = 0, exists = false ; i < 3 ; i++ )
	{
		if ( Yes )
		{
			if ( laststfile != NULLSTR )
			{
				if (  strcmp(statusfile, laststfile) == STREQUAL )
				{
					fd = SYSERROR;
					goto same;
				}

				free(laststfile);
			}

			laststfile = newstr(statusfile);
		}

		if ( (fd = open(statusfile, O_READ)) != SYSERROR )
		{
			if
			(
				read(fd, (char *)&NNstate, sizeof NNstate) == sizeof NNstate
				&&
				strcmp(NNstate.version, StreamSCCSID) == STREQUAL
			)
			{
				register Str_p	strp;
same:
				for ( strp = outStreams ; strp < &outStreams[NSTREAMS] ; strp++ )
					if
					(
						strp->str_id != '\0'
						&&
						strcmp(strp->str_id, commandfile) == STREQUAL
					)
					{
						if ( fd != SYSERROR )
							(void)close(fd);
						return true;
					}

				if ( fd != SYSERROR )
					(void)close(fd);
				return false;
			}

			exists = true;

			(void)close(fd);
		}

		(void)sleep(1);
	}

	if ( exists )
	{
		Warn("daemon status file version mismatch, re-compile");
		return true;
	}

	return false;
}



/*
**	Read HeaderFile, and extract details.
*/

bool
read_header()
{
	int		fd;
	HdrReason	hr;
	bool		val;
	char *		cp;
	Handler *	handler;
	Time_t		msg_date;
	struct stat	statb;

	Trace2(1, "read_header \"%s\"", HeaderFile);

	if ( (fd = open(HeaderFile, O_READ)) == SYSERROR )
	{
		if ( Warnings )
			SysWarn("Can't read \"%s\"", HeaderFile);

		return false;
	}

	if ( (hr = ReadHeader(fd)) != hr_ok )
	{
		if ( Warnings )
			Warn("Header error \"%s\"", HeaderReason(hr));

		(void)close(fd);
		return false;
	}

	if ( (cp = GetEnv(ENV_RETURNED)) != NULLSTR )
	{
		free(cp);
		return false;
	}

	if
	(
		RHandler != NULLSTR
		&&
		strcmp(HdrHandler, RHandler) != STREQUAL
	)
	{
		(void)close(fd);
		return false;
	}

	/*
	**	Fetch protcocol details if from Home_Address, or All.
	*/

	Local = (bool)(strcmp(HdrSource, Home_Address) == STREQUAL);

	if
	(
		All
		?
		(
			SourceNode != NULLSTR
			&&
			strccmp(HdrSource, SourceNode) != STREQUAL
		)
		:
		!Local
	)
	{
		(void)close(fd);
		return false;
	}

	if ( (handler = GetHandler(HdrHandler)) == (Handler *)0 )
		cp = "Message";
	else
		cp = handler->descrip;
	
	(void)fstat(fd, &statb);

	WaitTime = time((long *)0) - statb.st_mtime;

	if ( MessageDetails )
		msg_date = statb.st_mtime - atol(HdrTt);

	if ( HdrSubpt[0] == FTP )
		val = read_ftp(fd, cp);
	else
	{
		val = false;
		(void)close(fd);
	}

	if ( val == false )
	{
		if ( !All )
			return false;

		print_header();

		Fprintf
		(
			stdout,
			Verbose?FormatV:Format,
			cp,
			HdrSource,
			HdrHandler,
			HdrDest,
			MesgLength
		);
	}

	/*
	**	Print message header info if requested.
	*/

	if ( MessageDetails )
	{
		if ( HdrEnv[0] != '\0' )
			Fprintf
			(
				stdout,
				"%*sEnv=\"%s\"\n",
				MSGINDENT, "",
				ExpandString(HdrEnv, strlen(HdrEnv))
			);

		Fprintf
		(
			stdout,
			"%*sRoute=\"%s\"\n",
			MSGINDENT, "",
			ExpandString(HdrRoute, strlen(HdrRoute))
		);

		Fprintf
		(
			stdout,
			"%*sDate: %.15s\n",
			MSGINDENT, "",
			ctime(&msg_date)+4
		);
	}

	return true;
}



/*
**	Function called from "ReadCom" to process a command.
**	Remember last file name (containing header),
**	and all "unlink" file names.
*/

bool
read_com(name, base, range)
	char *	name;
	long	base;
	long	range;
{
	Trace4(2, "command \"%s %ld %ld\"", name, base, range);

	if ( range > 0 )
	{
		MesgLength += range;
		if ( MesgFile != NULLSTR )
			free(MesgFile);
		MesgFile = HeaderFile;
		HeaderFile = name;
		MesgDataSize = MesgHdrSize;
		MesgHdrSize = range;
	}
	else
	if ( base == 0 )
	{
		register UN_p	unp;

		if ( (unp = UnFreelist) == (UN_p)0 )
			unp = Talloc(UnlinkN);
		else
			UnFreelist = unp->un_next;

		unp->un_next = UnList;
		UnList = unp;
		unp->un_name = name;
	}
	else
	{
		Time_to_die = base;
		free(name);
	}

	return true;
}



/*
**	Read the data part of message for FTP header.
*/

bool
read_ftp(fd, type)
	int		fd;
	char *		type;
{
	FthReason	fthr;

	Trace5
	(
		2,
		"read_ftp type %s, datasize %ld, hdrsize %ld, datalength %ld",
		type,
		MesgDataSize,
		MesgHdrSize,
		DataLength
	);

	if ( DataLength == 0 )
	{
		(void)close(fd);

		if ( (fd = open(MesgFile, O_READ)) == SYSERROR )
		{
			if ( Warnings )
				SysWarn("Can't read message data from \"%s\"", MesgFile);

			return false;
		}
	}
	else
		MesgDataSize = DataLength;

	if ( (fthr = ReadFtHeader(fd, MesgDataSize, false)) != fth_ok )
	{
		if ( Warnings )
			Warn("FT Header \"%s\" error", FTHREASON(fthr));

		(void)close(fd);
		return false;
	}

	(void)close(fd);

	if ( !All && strcmp(FthFrom, Me.P_name) != STREQUAL )
		return false;

	/*
	**	Have valid FTP message to print
	*/

	print_header();

	if ( !Verbose )
	{
		Fprintf
		(
			stdout,
			Format,
			type,
			FthFrom,
			FthTo,
			HdrDest,
			MesgLength
		);
	}
	else
	{
		register FthFD_p	fp;

		Fprintf
		(
			stdout,
			FormatV,
			type,
			FthFrom,
			FthTo,
			HdrDest,
			MesgLength
		);

		if ( (fthr = GetFthFiles()) != fth_ok )
		{
			Warn("FT Header \"%s\" error", FTHREASON(fthr));
			return true;
		}

		for ( fp = FthFiles ; fp != (FthFD_p)0 ; fp = fp->f_next )
			Fprintf
			(
				stdout,
				"%27s %10ld  %.12s\n",
				fp->f_name,
				fp->f_length,
				ctime(&fp->f_time)+4
			);

		FreeFthFiles();
	}

	return true;
}



/*
**	Print out header on demand
*/

void
print_header()
{
	if ( !LinkPrinted )
	{
		LinkPrinted = true;
		Fprintf(stdout, LinkName);
	}
}



/*
**	Interrogate for action
*/

bool
query(timed_out)
	bool		timed_out;
{
	register char *	cp;
	register int	n;
	char		buf[4];
	static char *	yes = "yes";

	if ( timed_out )
	{
		Fprintf(stdout, "(TIMED OUT)\n");
		Fflush(stdout);
		return true;
	}

	Fprintf(stdout, "Stop ? (y or n) ");

	if ( Yes )
	{
		Fprintf(stdout, "%s\n", yes);
		Fflush(stdout);
		return true;
	}

	Fflush(stdout);

	for( cp = buf, n = 0 ; ; )
	{
		if ( read(0, cp, 1) <= 0 )
			finish(1);

		if ( *cp == '\n' )
		{
			*cp = '\0';
			break;
		}

		if ( ++n < sizeof buf )
			cp++;
	}

	if ( n > 0 && n <= strlen(yes) && strncmp(buf, yes, n) == STREQUAL )
		return true;

	return false;
}



/*
**	Give child correct privileges to run router.
*/

void
setup()
{
	Trace1(1, "setup()");

#	if	SYSTEM > 0
	if ( ulimit(2, ULIMIT) == SYSERROR )
		Syserror("ulimit(%ld)", ULIMIT);
#	endif

#ifndef	NICEDAEMON
#define	NICEDAEMON	0
#endif	NICEDAEMON

#	if	SYSTEM > 0
	(void)nice(NICEDAEMON-nice(0));
#	else	SYSTEM > 0
	if ( getuid() == 0 )
	{
		(void)nice(-40);
		(void)nice(20+NICEDAEMON);
	}
#	endif	SYSTEM > 0

	SetUser(ACSNETUID, ACSNETGID);

	(void)signal(SIGINT, SIG_IGN);
	(void)signal(SIGQUIT, SIG_IGN);
	(void)signal(SIGHUP, SIG_IGN);
}



extern char *	ExecutR();

/*
**	Return a non-local message to its origin.
*/

void
Return(link)
	char *	link;
{
	VarArgs	args;
	char *	errs;

	if ( (errs = GetEnv(ENV_NORET)) != NULLSTR )
	{
		free(errs);
		return;
	}

	FIRSTARG(&args) = RECEIVER;
	if ( Traceflag )
		NEXTARG(&args) = NumericArg('T', Traceflag);
	NEXTARG(&args) = concat("-h", HomeNode, NULLSTR);
	if ( Reason != NULLSTR )
		NEXTARG(&args) = concat("-rMessage on link ", link, " stopped by operator at ", Home_Address, "\n", Reason, NULLSTR);
	else
		NEXTARG(&args) = concat("-rMessage on link ", link, " stopped by operator at ", Home_Address, NULLSTR);
	NEXTARG(&args) = concat("-l", link, NULLSTR);
	NEXTARG(&args) = NumericArg('t', LastTime(HdrRoute) + WaitTime);
	NEXTARG(&args) = MesgFile;

	if ( (errs = ExecutR(&args, setup)) != NULLSTR )
	{
		Error(strlen(errs) > 1 ? errs : "message return failed");
		free(errs);
	}

	free(ARG(&args, 4));
	free(ARG(&args, 3));
	free(ARG(&args, 2));
	free(ARG(&args, 1));
}
