/*
**	Patterns and functions to call a host via a generic Virtual Circuit Device.
**
**	SCCSID %W% %E%
*/

#include	"global.h"
#include	"debug.h"

#include	"caller.h"

#define	MAXDEV	8			/* Maximum specifiable alternate devices */


char
	*CallArgs= CALLARGS,		/* optional args for call */
	*Daemon1 = NNDAEMON,		/* standard daemon */
	*Daemon2 = CNDAEMON,		/* alternate protocol */
	*Daemon3 = PNDAEMON,		/* alternate protocol */
	*HostFlg = "-BF";		/* batch mode, no fork */

char
	*VCReq	 = NULLSTR,		/* Request to set-up virtual circuit */
	*Device	 = NULLSTR,		/* Device for link */
	*Host	 = NULLSTR,		/* Who am i */
	*Passwd	 = NULLSTR,		/* Password at Target */
	*Retries = "3",			/* Number of retries */
	*Retry   = "30",		/* Retry sleep time */
	*Speed	 = NULLSTR,		/* Speed of link */
	*Target	 = NULLSTR,		/* Target host */
	*TimeOut = "15";		/* Timeout for open */

int
	DefFlag	 = '1',			/* Default daemon to start [1,2,3] */
	Devices  = 0,			/* Count of device names if separate "-d" args */
	Tcount	 = 0,			/* count of Timeouts */
	Ccount	 = 0,			/* count of VCD login attempts */
	Lcount	 = 0,			/* count of Unix login attempts */
	Wait	 = 4;			/* Minutes to wait after failure */

int
	AtVCD(),
	AtUNIX(),
	DefStart(),
	EchoOff(),
	FindEcho(),
	GotEof(),
	GotLogin(),
	GotPasswd(),
	GotTimeout(),
	NoShellFail(),
	Start1(),
	Start2(),
	Start3(),
	ToVCD(),
	ToUNIX(),
	UNIXfail();

/*
**	VCD messages (examples taken from X.25 PAD setup)
*/

char	Connect[]	= "^> com";
char	Disconn[]	= "clear cause";
char	Prompt[]	= "clr";
char	VCDEcho[]	= ".1.echo";
char	VCDNoEcho[]	= ".0.no echo";

/*
**	VCD commands (examples taken from X.25 PAD setup)
*/

char	ClearEcho[]	= "set?2:0\r";
char	ClearVCD[]	= "\r";
char	EnqEcho[]	= "par?2\r";

/*
**	Other common patterns
*/

char	CNstarts	= CNSTARTS;
char	DActive[]	= NOGOMSG;
char	DError		= "daemon: error";
char	Eof[]		= EOFSTR;
char	LoginI[]	= "[Ll]ogin +[Ii]ncorrect";
char	Login[]		= "[Ll]ogin";
char	NNstarts	= NNSTARTS;
char	NoShell		= "no shell";
char	PNstarts	= PNSTARTS;
char	Password[]	= "^[Pp]assword";
char	Timeout[]	= TIMEOUT;
char	WPassword[]	= "[Ww]rong +[Pp]assword";

/*
**	The pattern/action sets
*/

PatList	VCDnet[] =			/* At the Virtual Circuit Setup Device */
{
	{ Login, GotLogin },
	{ Connect, ToUNIX },
	{ Prompt, FindEcho },
	{ VCDEcho, EchoOff },
	{ VCDNoEcho, AtVCD },
	{ NNstarts, Start1 },
	{ CNstarts, Start2 },
	{ PNstarts, Start3 },
	{ DActive, UNIXfail },
	{ NoShell, UNIXfail },
	{ DError, UNIXfail },
	{ Timeout, GotTimeout },
	{ Eof, GotEof },
	{ 0, 0}
};

PatList UNIXnet[] =			/* Talking to UNIX login process */
{
	{ LoginI, ToVCD },
	{ WPassword, AtUNIX },
	{ Login, GotLogin },
	{ Password, GotPasswd },
	{ NNstarts, Start1 },
	{ CNstarts, Start2 },
	{ PNstarts, Start3 },
	{ DActive, UNIXfail },
	{ NoShell, UNIXfail },
	{ DError, UNIXfail },
	{ Disconn, ToVCD },
	{ Timeout, GotTimeout },
	{ Eof, GotEof },
	{ 0, 0 }
};



/*
**	Function to process arguments, either from NNcall, or from CALLARGS.
*/

void
Args(argc, argv)
	register int	argc;
	register char *	argv[];
{
	while ( --argc > 0 )
	{
		register int	c;

		if ( (c = Traceflag) >= 2 )
		{
			Traceflag = 0;
			Command("trace Args: \"", argv[1], "\"", NULLSTR);
			Traceflag = c;
		}

		if ( **++argv == '-' )
		{
			while ( c = *++*argv )
			{
				switch ( c )
				{
				case '1':
					if ( *++*argv )
						Daemon1 = *argv;
					DefFlag = c;
					goto break2;

				case '2':
					if ( *++*argv )
						Daemon2 = *argv;
					DefFlag = c;
					goto break2;

				case '3':
					if ( *++*argv )
						Daemon3 = *argv;
					DefFlag = c;
					goto break2;

				case 'R':
					Retries = ++*argv;
					goto break2;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) == 0 )
						Traceflag = 1;
					goto break2;

				case 'd':
					if ( Devices >= MAXDEV )
					{
						Command
						(
							"fail too many devices in ",
							CallArgs,
							NULLSTR
						);
						Exit(1);
					}
					if ( Devices++ == 0 )
						Device = ++*argv;
					else
						Device = concat(Device, " ", ++*argv, NULLSTR);
					goto break2;

				case 'l':
					Host = ++*argv;
					goto break2;

				case 'p':
					Passwd = ++*argv;
					goto break2;

				case 'r':
					Retry = ++*argv;
					goto break2;

				case 's':
					Speed = ++*argv;
					goto break2;

				case 't':
					TimeOut = ++*argv;
					goto break2;

				case 'v':
					VCReq = ++*argv;
					goto break2;

				case 'w':
					Wait = atoi(++*argv);
					break;
					
				default:
					Command
					(
						"fail unexpected flag \"-",
						*argv,
						"\" in ",
						CallArgs,
						NULLSTR
					);
					Exit(1);
				}
			}
break2:			;
		}
		else
			Target = *argv;
	}
}



/*
**	Called from "main()" in caller.c to initialise call.
*/

init(argc, argv)
	int		argc;
	char *		argv[];
{
	register char *	cp;

	Host = NodeName();

	Args(argc, argv);

	if ( Target != NULLSTR )
		CallArgs = concat(SPOOLDIR(), Target, "/", CallArgs, NULLSTR);
	else
	{
		if ( (cp = strrchr(*argv, '/')) != NULLSTR )
		{
			*cp = '\0';
			cp = *argv;
		}
		else
			cp = ".";

		CallArgs = concat(cp, "/", CallArgs, NULLSTR);
	}
	(void)readargs(CallArgs, Args);

	if
	(
		Target == NULLSTR
		||
		VCReq == NULLSTR
		||
		Device == NULLSTR
	)
	{
		Command("fail incomplete initialisation", NULLSTR);
		Exit(1);
	}

	Command("retry ", Retry, " ", Retries, NULLSTR);
	Command("timeout ", TimeOut, NULLSTR);
	Command("open ", Device, NULLSTR);
	if ( Speed != NULLSTR )
		Command("speed ", Speed, NULLSTR);
	Command("read", NULLSTR);

	ToVCD();
}



ToVCD()
{
	Tcount = 0;
	flushinput();
	Command("write ", ClearVCD, NULLSTR);
	state(VCDnet);
	reset();
}



AtVCD()
{
	if ( ++Ccount >= 7 )
	{
		Command("fail VCD setup failed", NULLSTR);
		Exit(1);
	}

	Command("write ", VCReq, "\r", NULLSTR);
	reset();
}



ToUNIX()
{
	Tcount = 0;
	Lcount = 0;

	state(UNIXnet);
	reset();
}



AtUNIX()
{
	if ( ++Lcount >= 7 )
	{
		Command("fail too many login attempts at ", Target, NULLSTR);
		Exit(1);
	}

	Command("sleep 10", NULLSTR);
	flushinput();
	Command("write \r", NULLSTR);
	reset();
}



GotLogin()
{
	if ( ++Lcount >= 7 )
	{
		Command("fail too many login attempts at ", Target, NULLSTR);
		Exit(1);
	}

	Command("write ", Host, "\r", NULLSTR);
	state(UNIXnet);
	reset();
}



GotPasswd()
{
	if ( Passwd == NULLSTR )
		Command("write \r", NULLSTR);
	else
		Command("write ", Passwd, "\r", NULLSTR);
	reset();
}



DefStart()
{
	switch ( DefFlag )
	{
	default:
	case '1':
		Start1();
		break;
	case '2':
		Start2();
		break;
	case '3':
		Start3();
		break;
	}
}



Start1()
{
	Command("daemon ", Daemon1, NULLSTR);
	Command("succeed ", HostFlg, " ", Target, NULLSTR);
	Exit(0);
}



Start2()
{
	Command("daemon ", Daemon2, NULLSTR);
	Command("succeed ", HostFlg, " ", Target, NULLSTR);
	Exit(0);
}



Start3()
{
	Command("daemon ", Daemon3, NULLSTR);
	Command("succeed ", HostFlg, " ", Target, NULLSTR);
	Exit(0);
}



Exit(n)
	int	n;
{
	sleep(2);
	exit(n);
}



GotTimeout()
{
	if ( ++Tcount >= 3 )
	{
		Command("fail TIMEOUT", NULLSTR);
		Exit(1);
	}

	if ( current == VCDnet )
		Command("write ", ClearVCD, NULLSTR);
	else
		Command("write @\r", NULLSTR);

	reset();
}



GotEof()
{
	Command("fail unexpected EOF", NULLSTR);
	Exit(1);
}



UNIXfail()
{
	Command("fail ", input, NULLSTR);
	Exit(1);
}


FindEcho()
{
	Command("write ", EnqEcho, NULLSTR);
	reset();
}


EchoOff()
{
	Command("write ", ClearEcho, NULLSTR);
	reset();
}
