/*
**	Written by Doug Kingston, Centrum voor Wiskunde en Informatica, NL
*/

#include "global.h"
#include <stdio.h>
#include <setjmp.h>
#include <signal.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/file.h>

int	Fd = (-1);
FILE	*ComFp;
int	cargc;
char	*cargv[16];
char	commandbuff[512];
int	forkoff = 0;
int	Traceflag = 0;
char	*hostdir = ".";
char	*script = "callscript";

char	*Name;

jmp_buf	env;

#define	C_CONNECT	1
#define	C_LABEL 	2
#define	C_GOTO		3
#define	C_EXPECT	4
#define	C_SEND		5
#define	C_EXEC		6
#define	C_WAIT		7
#define	C_LOCK		8
#define	C_CLOSE		9
#define	C_EXIT		10
#define C_OPEN		11
#define	C_UNKNOWN	999

struct command {
	char	*name;
	int	minargs;
	int	code;
} cmds[] = {
	"connect",	2,	C_CONNECT,
	"label",	2,	C_LABEL,
	"goto",		2,	C_GOTO,
	"expect",	3,	C_EXPECT,
	"send",		2,	C_SEND,
	"exec",		2,	C_EXEC,
	"wait",		2,	C_WAIT,
	"lock",		1,	C_LOCK,
	"close",	1,	C_CLOSE,
	"exit",		2,	C_EXIT,
	"open",		2,	C_OPEN,
	0,		0,	0
};

extern	char	*index(), *rindex();

main (argc, argv)
int	argc;
char	**argv;
{
	FILE	*fp;
	register int	i;

	for (i = 1; i < argc; i++) {
		if(*argv[i] != '-') {
			register char	*cp;

			script = argv[i];
			if (cp = rindex(script, '/')) {
				*cp = 0;
				hostdir = newstr(script);
				*cp = '/';
				if (Traceflag)
					fprintf(stderr, "Chdiring to '%s'.\n", hostdir);
				if (chdir (hostdir) < 0) {
					perror(hostdir);
					exit(9);
				}
			}
			break;
		}
		switch (argv[i][1]) {
		case '&':
			forkoff++;
			break;
		case 'T':
			Traceflag = 1;
			if (argv[i][2])
				Traceflag = atoi(&argv[i][2]);
			break;
		case 'h':
			/* Ignore */
			break;
		default:
			fprintf(stderr, "Unknown option '%s'\n", argv[i]);
			break;
		}
	}

	if ( forkoff ) {
		switch( fork() ) {
		case SYSERROR:
			Syserror("Can't fork");
			return 1;

		case 0:
			break;

		default:
			return 0;
		}

		(void)signal(SIGHUP, SIG_IGN);
		(void)signal(SIGINT, SIG_IGN);
		(void)signal(SIGQUIT, SIG_IGN);
	}

#	if	KILL_0 != 1
	(void)signal(SIG0, SIG_IGN);
#	endif	KILL_0 != 1

	if ( DaemonActive(".", true) ) {
		if (Traceflag)
			fprintf(stderr, "Daemon is already active.\n");
		exit(0);
	}
	
	(void)SetDaemonActive(".", getpid());

	if ((ComFp = fopen(script, "r")) == NULL) {
		perror(argv[1]);
		exit (98);
	}
	while (nextcommand() != NULL) {
		switch (commandtype()) {
		case C_CONNECT:
			if (Traceflag) {
				fprintf (stderr, "Connect to '%s' via Micom ...", cargv[1]);
				fflush(stderr);
			}
			Fd = openDEVICE(cargv[1]);
			if (Traceflag) {
				fprintf (stderr, " %s.\n",
					 Fd < 0 ? "failed" : "succeeded");
				fflush(stderr);
			}
			if(Fd < 0 && cargc > 2)
				gotolabel(cargv[2]);
			break;

		case C_OPEN:
			if (Traceflag) {
				fprintf (stderr, "Opening '%s' ...", cargv[1]);
				fflush(stderr);
			}
			Fd = open (cargv[1], 2);
			if (Traceflag) {
				fprintf (stderr, " %s.\n",
					 Fd < 0 ? "failed" : "succeeded");
				fflush(stderr);
			}
			if(Fd < 0 && cargc > 2)
				gotolabel(cargv[2]);
			break;

		case C_LABEL:
			break;		/* Ignore */

		case C_GOTO:
			gotolabel(cargv[1]);
			break;

		case C_EXPECT:
			if (!expect(cargv[1], cargv[2]))
				if(cargc > 3)
					gotolabel(cargv[3]);
			break;

		case C_SEND:
			send(cargv[1]);
			break;

		case C_EXEC:
			fclose (ComFp);
			dup2(Fd, 0);
			dup2(Fd, 1);
			close(Fd);
			setgid (ACSNETGID);
			setuid (ACSNETUID);
			execv(cargv[1], &cargv[1]);
			perror(cargv[1]);
			break;

		case C_WAIT:
			sleep(atoi(cargv[1]));
			break;

		case C_LOCK:
			if (Fd >= 0 && flock(Fd, LOCK_EX|LOCK_NB) == 0)
				if (cargc > 1)
					gotolabel(cargv[1]);
			break;

		case C_CLOSE:
			if (Fd >= 0)
				close(Fd);
			Fd = (-1);
			break;

		case C_EXIT:
			if (cargc > 2)
				fprintf(stderr, "Exiting: %s\n", cargv[2]);
			exit(atoi(cargv[1]));
			break;

		default:
			break;		/* Ignore */
		}
	}
	fclose(ComFp);
	fprintf(stderr, "End of script file reached\n");
	exit(1);
}

nextcommand()
{
	char	*cp;

playitagainsam:
	if (fgets(commandbuff, sizeof commandbuff, ComFp) == NULL)
		return(NULL);
	if (cp = index(commandbuff, '\n'))
		*cp = 0;
	if (Traceflag > 1)
		fprintf(stderr, "   Process '%s'\n", commandbuff);
	cp = commandbuff;
	while (isspace(*cp))
		cp++;
	cargc = 0;
	while (*cp) {
		if (*cp == '#') 
			break;		/* "<wsp>#", ignore rest of line */
		cargv[cargc] = cp;
		cargc++;
		cargv[cargc] = (char *)0;
		while(*cp && !isspace(*cp))
			cp++;
		*cp++ = 0;
		while (isspace(*cp))
			cp++;
	}
	if (cargc == 0)
		goto playitagainsam;
	return (cargc);
}

commandtype()
{
	struct command *cmdp = cmds;

	for(; cmdp->name != 0; cmdp++) {
		if (strcmp(cargv[0], cmdp->name) == 0)
			break;
	}
	if (cmdp->name == 0) {
		fprintf(stderr, "Unknown script command '%s'\n", cargv[0]);
		return (C_UNKNOWN);
	}
	if (cargc < cmdp->minargs) {
		fprintf(stderr, "Command %s must have at least %d args\n",
			cmdp->name, cmdp->minargs);
		return (C_UNKNOWN);
	}
	return (cmdp->code);
}

expect(pat, timeout)
char	*pat, *timeout;
{
	char buf[1024];
	register char *cp, *pp;
	int	len;

	int	alrmcatch();

	if (Traceflag) {
		fprintf (stderr, "Looking for '%s': ", pat);
		fflush(stderr);
	}

	if (setjmp(env)) {
		alarm(0);
		return(0);
	}
	signal (SIGALRM, alrmcatch);
	alarm(atoi(timeout));
	cp = buf;
	len = strlen(pat);
	while (read (Fd, cp, 1) > 0) {
		*cp &= 0177;
		if (Traceflag) {
			if (isprint(*cp) || isspace(*cp))
				fputc(*cp, stderr);
			else
				fprintf(stderr, "\\%03o", *cp);
		}
		fflush(stderr);
		*(++cp) = 0;
		if ((cp - buf) >= len && strcmp(cp - len, pat) == 0) {
			alarm(0);
			if (Traceflag) {
				fprintf (stderr, "\nGot it.\n");
				fflush(stderr);
			}
			return (1);
		}
	}
	alarm(0);
	return(0);
}

send(str)
char	*str;
{
	if (Traceflag) {
		fprintf (stderr, "Sending (%s) ...", str);
		fflush(stderr);
	}

	while (*str) {
		switch (*str) {
		case '\\':
			str++;
			switch (*str) {
			case '\\':
				write(Fd, str, 1);
				if (Traceflag)
					write(2, str, 1);
				break;
			case 'd':
				sleep(1);
				break;
			case 'n':
				write(Fd, "\n", 1);
				if (Traceflag)
					write(2, "\\n", 2);
				break;
			case 'r':
				write(Fd, "\r", 1);
				if (Traceflag)
					write(2, "\\r", 2);
				break;
			case 's':
				write(Fd, " ", 1);
				if (Traceflag)
					write(2, " ", 1);
				break;
			case 't':
				write(Fd, "\t", 1);
				if (Traceflag)
					write(2, "\\t", 2);
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			    {
				char	c;

				c = (*str) - '0';
				if (isdigit(*++str)) {
					c <<= 3;
					c |= (((*str) - '0') & 07);
					if (isdigit(*++str)) {
						c <<= 3;
						c |= (((*str) - '0') & 07);
					}
				}
				write (Fd, &c, 1);
				if (Traceflag) {
					fprintf(stderr, "\\%03o", c&0377);
					fflush(stderr);
				}
				break;
			    }
			default:
				write(Fd, str, 1);
				if (Traceflag)
					write(2, str, 1);
				break;
			}
			str++;
			break;
		default:
			write(Fd, str, 1);
			if (Traceflag)
				write(2, str, 1);
			str++;
		}
	}

	if (Traceflag) {
		fprintf (stderr, " sent.\n");
		fflush(stderr);
	}
}

gotolabel(label)
char	*label;
{
	char	temp[sizeof commandbuff];

	strcpy(temp, label);
	rewind(ComFp);
	while (nextcommand() != NULL) {
		if (commandtype(cargv[0]) == C_LABEL
		  && strcmp (cargv[1], temp) == 0)
			return;
	}
	fprintf(stderr, "Failed to find label '%s' in script.\n", temp);
}

alrmcatch()
{
	fprintf(stderr, "\n -- Timeout --\n");
	longjmp (env, 1);
}

finish(code)
{
	exit(code);
}
