#!/bin/sh

# acscallmgr
#	$Header: acscallmgr,v 1.3 86/12/10 13:37:36 brian Exp $
#
#	Shell script to manage ACSnet calls.
#	- With no argument, calls all those hosts that have executable
#	  call scripts in their spool directories.
#	- With an argument of "-retry", calls all those hosts that were
#	  busy last time round.
#	- With an argument of "host-name" calls that host.
#
#	Links that do not have executable call programs in their spool
#	directories will not get called, even if explicitly named.
#
#	Link monitoring:
#	If calls cannot get through for more than 1 day, a message will
#	be mailed to the 'acsnet' user warning of failure to contact the
#	site.
#
#	Link debugging:
#	- links that have a file named 'debug' in their spool directory
#	  will use the contents of the debug file as extra arguments to
#	  NNcall.  Usually debug contains '-T2' to debug the link.
#	- The debug output goes in a file called 'callog' and the output
#	  from the previous run goes in a file called 'callog.old'
#	- If a file called 'undebug' exists that has a date in the past,
#	  debugging is stopped by removing the debug file.  'Undebug' can
#	  be created with a modify date in the future with System V touch
#	  or through an at job.
#
#					Brian Coogan 30 Dec 85, Oct 86
#						    
#

# Callout lines to check for availability (must be in callargs too)
# If none of these lines are available, a message can be displayed on each line
# warning the users they're in the way. Most often this is caused by dialling
# up and leaving the terminal logged on.  Defining modems as null disables the
# message AND the check for modem availability.  Turn off retrymsg or
# domessage below to limit/disable warnings.
modems=ttyd0
# retrymsg=yes	# if this is non-null users will be warned on retries.
# domessage=yes	# if this is set, warnings will be done.


PATH=/usr/spool/ACSnet/_lib:$PATH;export PATH
whofile=/tmp/who.$$	# temp who file

cd /usr/spool/ACSnet
hosts="$@"

#
#	Lock-out: go away if a previous incarnation still exists
#
lock=_lib/acscallmgr.l
echo $$ >$lock.t
/etc/link $lock.t $lock		# nb: ln always succeeds for root!
if [ $? -ne 0 ]
then
	if kill -0 `cat $lock` 2>/dev/null
	then
		rm -f $lock.t
		exit 0
	else
		echo $$ >$lock
	fi
fi
rm -f $lock.t
trap "rm -f $lock; exit 1" 1 2 3 15

#
#	If no hosts were given or we're retrying, look at all the hosts
#	and work out which are callable (executable call files).
#	If retrying, only call the hosts that were busy.
#
if [ $# -eq 0 -o "$1" = "-retry" ]
then
	hosts=
	for h in [a-zA-Z]*
	do
		if [ ! -x $h/call ]
		then
			continue # permanent connection
		fi

		if [ "$1" = "-retry" -a -f $h/log ]
		then
			#
			#	Only call if last call failed busy
			#
			if	tail -1 $h/log |
				egrep "fail.*(RETRY)|locked \(UUCP\)" >/dev/null
			then
				hosts="$hosts$spc$h"
				spc=' '
			fi
		else
			hosts="$hosts$spc$h"
			spc=' '
		fi
	done
fi
if [ -z "$hosts" ]
then
	exit 0
fi

#
# If people are in our way on ALL of the modem lines, let them know.
#
who > $whofile
# Look for a free line
for tty in $modems
do
	if [ -z "`grep "$tty" $whofile`" ]
	then
		ttyfree=yes
		break
	fi
done
rm $whofile
if [ "$modems" != "" -a "$ttyfree" != yes ]
then
msg="\r\n
\07Message from `hostname`!ACSnet \07`date`\07\r
Call to '$hosts' delayed.  Please don't forget to log out\r
when you finish.  Thanks!\r\n\07"
	# Notify the perpetrators
	if [ -n "$domessage" -a \( -n "$retrymsg" -o "$1" != "-retry" \) ]
	then
		for tty in $modems
		do
			echo "$msg" > /dev/$tty
		done
	fi
	exit 1
fi


#
#	Now do the actual calling...
#
for host in $hosts
do
	if [ ! -d "${host}" ]
	then
		echo Illegal host "$host". 1>&2
		continue
	fi
	if [ ! -x $host/call ]
	then
		echo No call file for "$host". 1>&2
		continue
	fi
	if [ -f $host/debug ]
	then
		log=$host/callog
		mv $log $log.old 2>/dev/null
		if [ -s $host/debug ]
		then
			debug=`cat $host/debug`
		else
			debug=-T9
		fi

		# Stop debugging after cut off date.
		# undebug is created by 'touch mmddhhmm undebug' or at(1).
		if [ -f $host/undebug ]
		then
			> $log
			cd $host
			find ../$log -newer undebug -exec rm -f undebug debug \;
		fi
	else
		log=$host/log
		debug=
	fi

	##
	##	Debug...
	##
	if [ "$1" = "-retry" ]
	then
		echo "\r\nACSnet: `date +'%a %h %d %T'` - retrying $host.\r\n"
	else
		echo "\r\nACSnet: `date +'%a %h %d %T'` - calling $host.\r\n"
	fi

	NNcall $debug $host/call -l$log -h$host
	x=$?

	# Determination of whether we got thru or not is via the exit status.
	# this is more than a little simplistic.

	# If we got through, record the fact that we succeeded.
	if [ $x -eq 0 ]
	then
		# Record the success
		date > $host/.ok
		continue
	fi

	# It failed, save the failure message if it didn't go into the log...
	if [ "$log" != $host/log ]
	then
		tail -1 $log >> $host/log
	fi

	# Now check out how long we haven't been getting through for.
	old=`find $host/.ok -mtime +2 -print 2>/dev/null`
	if [ -n "$old" ]
	then
		mailx -s "ACSnet link to $host" acsnet <<-EOF
		ACSnet intermittent connection to $host has been failing
		for at least 2 days.  May indicate problems.

		Current date:			`date`
		Date of last successful dial: 	`cat $old`
		EOF
		/bin/mv -f $old $old.old # So we don't repeat the message
	fi
done
rm -f $lock
